#!/usr/bin/env python
#coding: utf-8


# morfologik.py - fast and simple interface to morfologik
# Copyright (C) Bohdan R. Rau 2012-2014 <ethanak@polip.com>
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General Public
# License along with this program.  If not, write to:
# 	The Free Software Foundation, Inc.,
# 	51 Franklin Street, Fifth Floor
# 	Boston, MA  02110-1301, USA.


"""
Interfejs do Morfologika
"""

from morfoll import *

class Morfologik(MorfoLL):
    
    @staticmethod
    def _lowerarg(arg):
        if isinstance(arg,str):
            arg=arg.decode('utf-8')
        else:
            arg=unicode(arg)
        return arg.lower().encode('utf-8')
    
    def Identify(self,word,strict=False):
        
        """
        Identyfikuje formę bazową oraz odmianę wyrazu.
        Zwraca listę krotek (pisownia,baza,id_bazy,bin_grama,str_grama)
        lub None, jeśli nie znaleziono słowa.
        Przyjmuje słowo w postaci stringu lub unikodu, wewnętrznie
        konwertuje je na string skonwertowany do małych liter.
        Znaczenie pól w krotce:
        0 - pisownia wyrazu jako string (może różnić się wielkością liter
            od wprowadzonego wyrazu)
        1 - pisownia wyrazu podstawowego jako string
        2 - wewnętrzny identyfikator wyrazu bazowego
        3 - zestaw znaczników czytelny dla maszyny (64-bit int)
        4 - zestaw znaczników czytelny dla człowieka
        """
        
        if strict:
	    tup=MorfoLL.IdentifyWordStd(self,self._lowerarg(word))
	else:
	    tup=MorfoLL.IdentifyWord(self,self._lowerarg(word))
        if not tup:
            return None
        rc=[]
        while tup:
            vid=tup[0]
            rc.append(tup[1:]+(DecodeGrama(tup[4]),))
            tup=MorfoLL.IdentifyNext(self,vid)
        return rc
    
    def Generate(self,baseword=None,baseid=None,filtr=None):
        
        """
        Generuje listę odmiany dla słowa bazowego podanego w postaci
        stringu lub identyfikatora otrzymanego z funkcji Identify.
        Możliwe jest filtrowanie za pomocą podania żądanych form
        gramatycznych.
        Zwracana jest lista krotek (pisownia,bin_grama,str_grama)
        lub None jeśli nie znaleziono.
        konwertuje je na string skonwertowany do małych liter.
        Znaczenie pól w krotce:
        0 - pisownia wyrazu jako string (może różnić się wielkością liter
            od wprowadzonego wyrazu)
        1 - zestaw znaczników czytelny dla maszyny (64-bit int)
        2 - zestaw znaczników czytelny dla człowieka
        """
        
        if baseword is None and baseid is None:
            raise Exception('baseword i baseid wykluczają się wzajemnie')
        if not filtr:
            filtr=0
        elif not isinstance(filtr,int):
            filtr=ParseGrama(str(filtr))
        if baseid is not None:
            tup=MorfoLL.GenWordById(self,baseid,filtr)
        else:
            tup=MorfoLL.GenWord(self,self._lowerarg(baseword),filtr)
        if not tup:
            return None
        rc=[]
        while tup:
            vid=tup[0]
            rc.append(tup[1:]+(DecodeGrama(tup[2]),))
            tup=MorfoLL.GenNext(self,vid,filtr)
        return rc
    
    @staticmethod
    def _rcsort(a,b):
        if a[0] < b[0]:
            return -1
        elif a[0] > b[0]:
            return 1
        elif a[1] > b[1]:
            return 1
        elif a[1] < b[1]:
            return -1
        return 0
    
    def GenerateAll(self,word,filtr=None):
        """
        Generuje listę odmiany dla dowolnego słowa. W przypadku
        jeśli podane słowo pasuje do więcej niż jednego słowa
        bazowego, lista zawierać będzie wszystkie możliwości.
        Możliwe jest filtrowanie za pomocą podania żądanych form
        gramatycznych.
        Zwracana jest lista krotek (pisownia,bin_grama,str_grama)
        lub None jeśli nie znaleziono.
        konwertuje je na string skonwertowany do małych liter.
        Znaczenie pól w krotce:
        0 - pisownia wyrazu jako string (może różnić się wielkością liter
            od wprowadzonego wyrazu)
        1 - zestaw znaczników czytelny dla maszyny (64-bit int)
        2 - zestaw znaczników czytelny dla człowieka
        """

        idset=set()
        a=self.Identify(word)
        if not a:
            return None
        rc=[]
        for w in a:
            if w[2] in idset:
                continue
            idset.add(w[2])
            b = self.Generate(baseid=w[2],filtr=filtr)
            if b:
                rc = rc +b
        if not rc:
            return None
        rc.sort(self._rcsort)
        pos=0
        while pos < len(rc)-1:
            if self._rcsort(rc[pos],rc[pos+1]) == 0:
                rc.pop(pos+1)
            else:
                pos += 1
        return rc
    
if __name__ == '__main__':
    import sys
    if len(sys.argv) < 2:
        print "Sposób użycia: %s słowo [grama]" % sys.argv[0]
        sys.exit(1)
        
    a=Morfologik()
    word = sys.argv[1]
    grama =sys.argv[2] if len(sys.argv) > 2 else None
    print "Rozpoznanie:"
    b=a.Identify(word)
    if not b:
        print "Nie rozpoznano"
        sys.exit(0)
    for w in b:
        print "%s\t%s\t%s" % (w[0],w[1],w[4])
        
    print "\nOdmiana:"
    b=a.GenerateAll(word,grama)
    if not b:
        print "Brak możliwości"
    else:
        for w in a.GenerateAll(word,grama):
            print "%s\t%s" % (w[0],w[2])
    
