/*
 * pymorfol.c - fast and simple Python interface to morfologik
 * Copyright (C) Bohdan R. Rau 2012 <ethanak@polip.com>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program.  If not, write to:
 * 	The Free Software Foundation, Inc.,
 * 	51 Franklin Street, Fifth Floor
 * 	Boston, MA  02110-1301, USA.
 */

#include <Python.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <libmorfologik.h>


static struct morfologik_data *md;

static struct {
    char *name;
    u_int64_t value;
} _consts[]={
    {"WT_adj",WT_adj},
    {"WT_adjp",WT_adjp},
    {"WT_adjc",WT_adjc},
    {"WT_adv",WT_adv},
    {"WT_conj",WT_conj},
    {"WT_num",WT_num},
    {"WT_pact",WT_pact},
    {"WT_pant",WT_pant},
    {"WT_pcon",WT_pcon},
    {"WT_ppas",WT_ppas},
    {"WT_ppron12",WT_ppron12},
    {"WT_ppron3",WT_ppron3},
    {"WT_pred",WT_pred},
    {"WT_prep",WT_prep},
    {"WT_siebie",WT_siebie},
    {"WT_subst",WT_subst},
    {"WT_verb",WT_verb},
    {"WT_brev",WT_brev},
    {"WT_interj",WT_interj},
    {"WT_xxx",WT_xxx},
    {"WT_nie",WT_nie},
    {"WT_advp",WT_advp},

    {"WM_sg",WM_sg},
    {"WM_pl",WM_pl},

    {"WM_NUM_MASK",WM_NUM_MASK},

    {"WM_pred",WM_pred},

    {"WM_nom",WM_nom},
    {"WM_gen",WM_gen},
    {"WM_dat",WM_dat},
    {"WM_acc",WM_acc},
    {"WM_inst",WM_inst},
    {"WM_loc",WM_loc},
    {"WM_voc",WM_voc},

    {"WM_CASU_MASK",WM_CASU_MASK},

    {"WM_pos",WM_pos},
    {"WM_comp",WM_comp},
    {"WM_sup",WM_sup},

    {"WM_GRAD_MASK",WM_GRAD_MASK},

    {"WM_m1",WM_m1},
    {"WM_m2",WM_m2},
    {"WM_m3",WM_m3},
    {"WM_m",WM_m1 | WM_m2 | WM_m3},
    {"WM_f",WM_f},
    {"WM_n1",WM_n1},
    {"WM_n2",WM_n2},
    {"WM_n",WM_n1 | WM_n2},
    {"WM_p1",WM_p1},
    {"WM_p2",WM_p2},
    {"WM_p3",WM_p3},
    {"WM_p",WM_p1 | WM_p2 | WM_p3},

    {"WM_GENR_MASK",WM_GENR_MASK},

    {"WM_pri",WM_pri},
    {"WM_sec",WM_sec},
    {"WM_tri",WM_tri},

    {"WM_PERS_MASK",WM_PERS_MASK},

    {"WM_aff",WM_aff},
    {"WM_neg",WM_neg},
    {"WM_perf",WM_perf},
    {"WM_imperf",WM_imperf},
    {"WM_nakc",WM_nakc},
    {"WM_akc",WM_akc},
    {"WM_praep",WM_praep},
    {"WM_npraep",WM_npraep},
    {"WM_imps",WM_imps},
    {"WM_impt",WM_impt},
    {"WM_inf",WM_inf},
    {"WM_fin",WM_fin},
    {"WM_praet",WM_praet},
    {"WM_pot",WM_pot},
    {"WM_nstd",WM_nstd},
    {"WM_pun",WM_pun},
    {"WM_npun",WM_npun},
    {"WM_rec",WM_rec},
    {"WM_congr",WM_congr},

    {"WM_winien",WM_winien},
    {"WM_bedzie",WM_bedzie},

    {"WM_refl",WM_refl},
    {"WM_ger",WM_ger},
    {"WM_depr",WM_depr},
    {"WM_vulgar",WM_vulgar},
    {"WM_illegal",WM_illegal},
    {"WM_nonrefl",WM_nonrefl},
    {"WM_wok",WM_wok},
    {"WM_nwok",WM_nwok},
    {"WM_cmplx",WM_cmplx},
    {"WM_str3",WM_str3},

    {"WM_SEQ_FIRST",WM_SEQ_FIRST},
    {"WM_SEQ_END",WM_SEQ_END},
    {"WM_MASK",(1LL << WM_SEQ_END)-1},
    {"WT_SHIFT",WT_SHIFT},
    {"WT_MASK",WT_MASK},
    {NULL,0}
};

typedef struct {
    PyObject_HEAD;
} morfologik_MorfObject;

static int MorfObject_init(morfologik_MorfObject *self,
                        PyObject * args, PyObject *kwds)
{
    if (!md) {
        md=morfologik_InitSHM();
        if (!md) md=morfologik_Init(NULL);
        if (!md) {
		PyErr_SetString(PyExc_RuntimeError,"Błąd inicjalizacji morfologik");
		return -1;
	}
    }
    return 0;
}

static void MorfObject_dealloc(morfologik_MorfObject *self)
{
    self->ob_type->tp_free((PyObject*)self);
}

static PyObject *MO_parseGrama(PyObject *self, PyObject *args)
{
    char *s;
    if (!PyArg_ParseTuple(args,"s",&s)) {
	return NULL;
    }
    u_int64_t grama=morfologik_ParseGrama(s);
    if (!grama) {
	PyErr_SetString(PyExc_RuntimeError,"Błędna wartość parametru");
	return NULL;
    }
    return Py_BuildValue("L",grama);
}

static PyObject *MO_decodeGrama(PyObject *self, PyObject *args)
{
    u_int64_t grama;
    if (!PyArg_ParseTuple(args,"L",&grama)) {
	return NULL;
    }
    char buf[256];
    morfologik_DecodeGrama(grama,buf);
    return Py_BuildValue("s",buf);
}

static PyObject *MO_filterGrama(PyObject *self, PyObject *args)
{
    u_int64_t grama;u_int64_t filter;
    if (!PyArg_ParseTuple(args,"LL",&grama,&filter)) {
	return NULL;
    }
    return Py_BuildValue("i",morfologik_FilterGrama(grama,filter));
}

static PyObject *_cv_word(int id,char *pisownia,char *baza,int baseid,u_int64_t grama)
{
    if (id < 0) Py_RETURN_NONE;
    return Py_BuildValue("(issiL)",id,pisownia,baza,baseid,grama);
}

static PyObject *MO_identifyWord(PyObject *self, PyObject *args)
{
    char *s;
    if (!PyArg_ParseTuple(args,"s",&s)) {
	return NULL;
    }
    char *pisownia;
    char *baza;
    int baseid;
    int nextid;
    u_int64_t grama;
    nextid = morfologik_IdentifyWord(md,s,&pisownia,&baza,&baseid,&grama);
    return _cv_word(nextid,pisownia,baza,baseid,grama);
}

static PyObject *MO_identifyWordStd(PyObject *self, PyObject *args)
{
    char *s;
    if (!PyArg_ParseTuple(args,"s",&s)) {
	return NULL;
    }
    char *pisownia;
    char *baza;
    int baseid;
    int nextid;
    u_int64_t grama;
    nextid = morfologik_IdentifyWordStd(md,s,&pisownia,&baza,&baseid,&grama);
    return _cv_word(nextid,pisownia,baza,baseid,grama);
}

/*
static PyObject *MO_identifyWordC(PyObject *self, PyObject *args)
{
    char *s;
    if (!PyArg_ParseTuple(args,"s",&s)) {
	return NULL;
    }
    char *pisownia;
    char *baza;
    int baseid;
    int nextid;
    u_int64_t grama;
    nextid = morfologik_IdentifyWordC(md,s,&pisownia,&baza,&baseid,&grama);
    return _cv_word(nextid,pisownia,baza,baseid,grama);
}
*/

static PyObject *MO_identifyNext(PyObject *self, PyObject *args)
{
    int id;
    if (!PyArg_ParseTuple(args,"i",&id)) {
	return NULL;
    }
    char *pisownia;
    char *baza;
    int baseid;
    int nextid;
    u_int64_t grama;
    nextid = morfologik_IdentifyNext(md,id,&pisownia,&baza,&baseid,&grama);
    return _cv_word(nextid,pisownia,baza,baseid,grama);
}


static PyObject *_cg_word(int nextid,char * pisownia,u_int64_t grama)
{
    if (nextid < 0) Py_RETURN_NONE;
    return Py_BuildValue("(isL)",nextid,pisownia,grama);
}

static PyObject *MO_genWord(PyObject *self,PyObject *args)
{
    u_int64_t filter=0;
    char *baseword;
    char *pisownia;
    u_int64_t grama;
    int nextid;
    if (!PyArg_ParseTuple(args,"s|L",&baseword,&filter)) {
	return NULL;
    }
    nextid=morfologik_GenWord(md,baseword,&pisownia,&grama,filter);
    return _cg_word(nextid,pisownia,grama);
}

static PyObject *MO_genWordById(PyObject *self,PyObject *args)
{
    u_int64_t filter=0;
    int id;
    char *pisownia;
    u_int64_t grama;
    int nextid;
    if (!PyArg_ParseTuple(args,"i|L",&id,&filter)) {
	return NULL;
    }
    nextid=morfologik_GenWordById(md,id,&pisownia,&grama,filter);
    return _cg_word(nextid,pisownia,grama);
}

static PyObject *MO_WT_GET(PyObject *self,PyObject *args)
{
    u_int64_t grama;
    if (!PyArg_ParseTuple(args,"L",&grama)) {
	return NULL;
    }
    return Py_BuildValue("i",WT_GET(grama));
}

static PyObject *MO_WT_SET(PyObject *self,PyObject *args)
{
    u_int64_t grama;
    int type;
    if (!PyArg_ParseTuple(args,"Li",&grama,&type)) {
	return NULL;
    }
    return Py_BuildValue("L",WT_SET(grama,type));
}

static PyObject *MO_genNext(PyObject *self,PyObject *args)
{
    u_int64_t filter=0;
    int id;
    char *pisownia;
    u_int64_t grama;
    int nextid;
    if (!PyArg_ParseTuple(args,"i|L",&id,&filter)) {
	return NULL;
    }
    nextid=morfologik_GenNext(md,id,&pisownia,&grama,filter);
    return _cg_word(nextid,pisownia,grama);
}

static PyMethodDef morfoll_methods[] = {
    {"ParseGrama", (PyCFunction)MO_parseGrama, METH_VARARGS,
     "ParseGrama(string) zwraca int odpowiadający sparsowanym znacznikom.\nZnaczniki mogą być oddzielone kropkami lub dwukropkami.\n"},
    {"DecodeGrama", (PyCFunction)MO_decodeGrama, METH_VARARGS,
     "DecodeGrama(grama) zwraca string czytelny dla człowieka ze znacznikami\nodpowiadający binarnej postaci grama.\n"},
    {"FilterGrama", (PyCFunction)MO_filterGrama, METH_VARARGS,
     "FilterGrama(grama,filter) dopasowuje filtr do gramy"},
    {"WT_GET", (PyCFunction)MO_WT_GET, METH_VARARGS,
     "WT_GET(grama) pobiera typ wyrazu (część mowy) z grama"},
    {"WT_SET", (PyCFunction)MO_WT_SET, METH_VARARGS,
     "WT_SET(grama,typ) zwraca grama z ustawionym nowym typem"},
    {NULL,NULL, 0,NULL}
};

static PyMethodDef MorfObject_methods[] = {
    {"ParseGrama", (PyCFunction)MO_parseGrama, METH_VARARGS,
     "ParseGrama(string) zwraca int odpowiadający sparsowanym znacznikom.\nZnaczniki mogą być oddzielone kropkami lub dwukropkami.\n"},
    {"DecodeGrama", (PyCFunction)MO_decodeGrama, METH_VARARGS,
     "DecodeGrama(grama) zwraca string czytelny dla człowieka ze znacznikami\nodpowiadający binarnej postaci grama.\n"},
    {"FilterGrama", (PyCFunction)MO_filterGrama, METH_VARARGS,
     "FilterGrama(grama,filter) dopasowuje filtr do gramy"},
    {"IdentifyWord", (PyCFunction)MO_identifyWord, METH_VARARGS,
     "IdentifyWord(word) identyfikuje słowo. Zwraca None lub krotkę:\n\
0 - id dla funkcji IdentifyNext (int)\n\
1 - pisownia słowa (str)\n\
2 - pisownia słowa bazowego (str)\n\
3 - id słowa bazowego dla GenWordById(int)\n\
4 - zestaw znaczników odpowiadający formie gramatycznej wyrazu (int).\n"},
    {"IdentifyWordStd", (PyCFunction)MO_identifyWordStd, METH_VARARGS,
     "IdentifyWordStd(word) identyfikuje słowo\nbez prób składania poza przeczeniami.\n\
Zwraca None lub krotkę:\n\
0 - id dla funkcji IdentifyNext (int)\n\
1 - pisownia słowa (str)\n\
2 - pisownia słowa bazowego (str)\n\
3 - id słowa bazowego dla GenWordById(int)\n\
4 - zestaw znaczników odpowiadający formie gramatycznej wyrazu (int).\n"},
/*
    {"IdentifyWordC", (PyCFunction)MO_identifyWordC, METH_VARARGS,
     "IdentifyWordStd(word) identyfikuje słowo\nbez prób składania anachronizmów.\n\
Zwraca None lub krotkę:\n\
0 - id dla funkcji IdentifyNext (int)\n\
1 - pisownia słowa (str)\n\
2 - pisownia słowa bazowego (str)\n\
3 - id słowa bazowego dla GenWordById(int)\n\
4 - zestaw znaczników odpowiadający formie gramatycznej wyrazu (int).\n"},
*/
    {"IdentifyNext", (PyCFunction)MO_identifyNext, METH_VARARGS,
     "IdentifyNext(id) pobiera następne słowo. Zwraca None lub krotkę:\n\
0 - id dla funkcji IdentifyNext (int)\n\
1 - pisownia słowa (str)\n\
2 - pisownia słowa bazowego (str)\n\
3 - id słowa bazowego (int)\n\
4 - zestaw znaczników odpowiadający formie gramatycznej wyrazu (int).\n"},

    {"GenWord", (PyCFunction)MO_genWord, METH_VARARGS,
     "genWord(baseword,filter=0) generuje słowo na podstawie podanego słowa bazowego.\n\
Możliwe jest ograniczenie znalezionych słow do form gramatycznych pasujących do filtra.\n\
Zwraca None jeśli nie znajdzie słowa lub krotkę:\n\
0 - id dla funkcji GenNext (int)\n\
1 - pisownia słowa (str)\n\
3 - zestaw znaczników odpowiadający formie gramatycznej słowa (int).\n"},
    {"GenWordById", (PyCFunction)MO_genWordById, METH_VARARGS,
     "genWordBy(id) generuje słowo na podstawie podanego ID słowa bazowego.\n\
Możliwe jest ograniczenie znalezionych słow do form gramatycznych pasujących do filtra.\n\
Zwraca None jeśli nie znajdzie słowa lub krotkę:\n\
0 - id dla funkcji GenNext (int)\n\
1 - pisownia słowa (str)\n\
3 - zestaw znaczników odpowiadający formie gramatycznej słowa (int).\n"},
    {"GenNext", (PyCFunction)MO_genNext, METH_VARARGS,
     "genNext(id) pobiera następne słowo na podstawie id z GenWord/GenWordById.\n\
Możliwe jest ograniczenie znalezionych słow do form gramatycznych pasujących do filtra.\n\
Zwraca None jeśli nie znajdzie słowa lub krotkę:\n\
0 - id dla funkcji GenNext (int)\n\
1 - pisownia słowa (str)\n\
3 - zestaw znaczników odpowiadający formie gramatycznej słowa (int).\n"},
    {NULL, NULL, 0, NULL}           /* sentinel */
}; 

static PyTypeObject morfologik_MorfObjectType = {
    PyObject_HEAD_INIT(NULL)
    0,                         /*ob_size*/
    "morfoll.MorfoLL",             /*tp_name*/
    sizeof(morfologik_MorfObject), /*tp_basicsize*/
    0,                         /*tp_itemsize*/
    (destructor)MorfObject_dealloc,                         /*tp_dealloc*/
    0,                         /*tp_print*/
    0, /*tp_getattr*/
    0,                         /*tp_setattr*/
    0,                         /*tp_compare*/
    0,                         /*tp_repr*/
    0,                         /*tp_as_number*/
    0,                         /*tp_as_sequence*/
    0,                         /*tp_as_mapping*/
    0,                         /*tp_hash */
    0,                         /*tp_call*/
    0,                         /*tp_str*/
    0,                         /*tp_getattro*/
    0,                         /*tp_setattro*/
    0,                         /*tp_as_buffer*/
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,        /*tp_flags*/
    "Klasa obsługi Moefologika\n",           /* tp_doc */
    0,		               /* tp_traverse */
    0,		               /* tp_clear */
    0,		               /* tp_richcompare */
    0,		               /* tp_weaklistoffset */
    0,		               /* tp_iter */
    0,		               /* tp_iternext */
    MorfObject_methods,         /* tp_methods */
    0,                         /* tp_members */
    0,                         /* tp_getset */
    0,                         /* tp_base */
    0,                         /* tp_dict */
    0,                         /* tp_descr_get */
    0,                         /* tp_descr_set */
    0,                         /* tp_dictoffset */
    (initproc)MorfObject_init,  /* tp_init */
};

#ifndef PyMODINIT_FUNC	/* declarations for DLL import/export */
#define PyMODINIT_FUNC void
#endif
PyMODINIT_FUNC
initmorfoll(void) 
{
    PyObject* m;
    int i;

    morfologik_MorfObjectType.tp_new = PyType_GenericNew;
    if (PyType_Ready(&morfologik_MorfObjectType) < 0)
        return;

    m = Py_InitModule3("morfoll", morfoll_methods,
                       "Interfejs niskiego poziomu do Morfologika");
    Py_INCREF(&morfologik_MorfObjectType);
    PyModule_AddObject(m, "MorfoLL", (PyObject *)&morfologik_MorfObjectType);
    for (i=0;_consts[i].name;i++) {
        PyModule_AddObject(m,_consts[i].name,Py_BuildValue("L",_consts[i].value));
    }
}
