/*
 * File:   equalizer.c
 * Author: Marcin Kwiatkowski
 *
 * Created on September 22, 2010, 11:09 AM
 */

#include <stdlib.h>
#include <math.h>
#include <string.h>
#include "equalizer.h"


static double vsa = (1.0 / 4294967295.0); // Very small amount (Denormal Fix)


// ---------------
//| Initialise EQ |
// ---------------

// Recommended frequencies are ...
//
//  lowfreq  = 880  Hz
//  highfreq = 5000 Hz
//
// Set mixfreq to whatever rate your system is using (eg 8KHz)

void init_equaliser(struct EQSTATE* es, int lowfreq, int highfreq, int mixfreq) {
    // Clear state

    memset(es, 0, sizeof (struct EQSTATE));

    // Set Low/Mid/High gains to unity

    es->lg = 1.0;
    es->mg = 1.0;
    es->hg = 1.0;

    // Calculate filter cutoff frequencies

    es->lf = 2 * sin(M_PI * ((double) lowfreq / (double) mixfreq));
    es->hf = 2 * sin(M_PI * ((double) highfreq / (double) mixfreq));
}


// ---------------
//| EQ one sample |
// ---------------

// - sample can be any range you like :)
//
// Note that the output will depend on the gain settings for each band
// (especially the bass) so may require clipping before output, but you
// knew that anyway :)

double equalise(struct EQSTATE* es, double sample) {
    // Locals

    double l, m, h; // Low / Mid / High - Sample Values

    // Filter #1 (lowpass)

    es->f1p0 += (es->lf * (sample - es->f1p0)) + vsa;
    es->f1p1 += (es->lf * (es->f1p0 - es->f1p1));
    es->f1p2 += (es->lf * (es->f1p1 - es->f1p2));
    es->f1p3 += (es->lf * (es->f1p2 - es->f1p3));

    l = es->f1p3;

    // Filter #2 (highpass)

    es->f2p0 += (es->hf * (sample - es->f2p0)) + vsa;
    es->f2p1 += (es->hf * (es->f2p0 - es->f2p1));
    es->f2p2 += (es->hf * (es->f2p1 - es->f2p2));
    es->f2p3 += (es->hf * (es->f2p2 - es->f2p3));

    h = es->sdm3 - es->f2p3;

    // Calculate midrange (signal - (low + high))

    m = es->sdm3 - (h + l);

    // Scale, Combine and store

    l *= es->lg;
    m *= es->mg;
    h *= es->hg;

    // Shuffle history buffer

    es->sdm3 = es->sdm2;
    es->sdm2 = es->sdm1;
    es->sdm1 = sample;

    // Return result

    return (l + m + h);
}


// Usage sample
#if 0

#include "equaliser.h"

int main()
  {
        EQSTATE eq;

        // low band = 0-300Hz
        // mid band = 300-2400Hz
        // hi band = 2400-4000Hz
        // sampling rate 8000
        init_equaliser(&eq,300,2400,8000);

        // Set gains:
        eq.lg = 0.75;   // Cut low by 25%
        eq.mg = 1;      // leave as-is
        eq.hg = 1.1;    // gain high by 10%

        double waveform[32000];

        //presume we have waveform in double waveform[32000]; (32k samples -> 4 seconds)
        for(int idx=0;idx<32000;idx++)
            waveform[idx]=equalise(&eq,waveform[idx]);
  }

#endif
