/*
 * pyspda.c - Speech-Dispatcher python audio output
 * Copyright (C) Bohdan R. Rau 2011-2012 <ethanak@polip.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program.  If not, write to:
 * 	The Free Software Foundation, Inc.,
 * 	51 Franklin Street, Fifth Floor
 * 	Boston, MA  02110-1301, USA.
 */

#include <Python.h>
#include <alsa/asoundlib.h>
#include "config.h"
#ifdef HAVE_PULSE
#include <pulse/simple.h>
#include <pulse/error.h>
#endif
#include <sndfile.h>
#include <math.h>
#include <signal.h>
#include "sonic.h"
#include "equalizer.h"

#define PSD_AMODE_AO 1
#define PSD_AMODE_PULSE 2
#define PSD_AMODE_ALSA 3

static struct {
    char *name;
    int value;
} pyspda_consts[]={
	//{"AUDIO_AO",PSD_AMODE_AO},
	{"AUDIO_PULSE",PSD_AMODE_PULSE},
	{"AUDIO_ALSA",PSD_AMODE_ALSA},
	{NULL,0}
};

static int ensure(int v,int mini,int maxi)
{
    return (v<mini)?mini:(v>maxi)?maxi:v;
}


static int ao_initialized;

static int _allow_signal_catch=0;
static int _signal_catched;
static int _signal_got;

static void *_origin_signal;

void catch_keyboard(int dummy)
{
    _signal_got=1;
}

void catch_signal(void)
{
    _signal_got=0;
    if (!_allow_signal_catch) return;
    if (_signal_catched) return;
    _signal_catched=1;
    _origin_signal=signal(SIGINT,(void *)catch_keyboard);
}

void uncatch_signal(void)
{
    if (!_signal_catched) return;
    _signal_catched=0;
    signal(SIGINT,_origin_signal);
}

typedef struct {
	PyObject_HEAD;
	int audio_mode;
	int stop_playing;
	int contrast;
	double d_contrast;
	int volume;
	double d_volume;
	int samples_sent;
	int fragment_size;
        int eqenabled;
        double elg,emg,ehg,edivisor;
        struct EQSTATE es[2];
        int e_channels,e_samplerate,tc;
        


#if 0
	// ao part

	int ao_playing;
	ao_device *ao_device;
	int ao_driver;
	ao_sample_format ao_format;
	char *ao_driver_name;
	int ao_driver_set;
	struct ao_option *ao_options;
	int ao_frames;
#endif

	int alsa_playing;
	snd_pcm_t *alsa_pcm_handle;
	char *alsa_device_name;
	int alsa_channels;
	int alsa_rate;
	char *alsa_buffer;
	int alsa_buff_size;
	int alsa_buffer_pos;
	snd_pcm_uframes_t alsa_frames;
	
	
#ifdef HAVE_PULSE
	// pulse part

	int p_current_rate;
	pa_simple *p_simple;
	char *p_server;
	char *p_name;
	int p_server_set;
	int pulse_use_drain;
	int pulse_margin;
	int p_channels;
#endif

} pyspda_PySpdaObject;

static int PySpda_init(pyspda_PySpdaObject *self,
			PyObject * args, PyObject *kwds)
{


	int audio_mode;
	char *driver=NULL,*param=NULL,*name=NULL;
	static char *kwlist[]={"method","driver","param","name",NULL};

	//first argument must be one of PSD_AMODE

#ifdef HAVE_PULSE
	self->p_server=NULL;
#endif
	self->samples_sent=0;
	self->volume=100;
	self->contrast=0;
	self->fragment_size=128;
        self->elg = self ->emg = self ->ehg = self->edivisor = 1.0;

	if (! PyArg_ParseTupleAndKeywords(args, kwds, "i|sss", kwlist,
		&audio_mode,
		&driver,
		&param,
		&name)) return -1;

	if (audio_mode != PSD_AMODE_ALSA
#ifdef HAVE_PULSE	    
	    && audio_mode != PSD_AMODE_PULSE
#endif	    
	    ) {
		PyErr_SetString(PyExc_RuntimeError,"Illegal Audio Mode");
		return -1;
	}
	self->audio_mode=audio_mode;
	if (audio_mode ==PSD_AMODE_ALSA) {
		self->alsa_playing=0;
		if (param && strcmp(param,"default")) {
			self->alsa_device_name=strdup(param);
		}
		else {
			self->alsa_device_name=strdup("default");
		}
		self->alsa_buff_size=0;
		self->alsa_buffer=NULL;
		self->alsa_channels=0;
		self->alsa_rate=0;
	}
		
		
#if 0	
	else if (audio_mode == PSD_AMODE_AO) {
		self->ao_options=NULL;
		self->ao_playing=0;
		memset(&self->ao_format,0,sizeof(ao_sample_format));
		if (!ao_initialized) {
			ao_initialize();
			ao_initialized=1;
		}
		if (driver && strcmp(driver,"default")) {
			self->ao_driver=ao_driver_id(driver);
			if (self->ao_driver >= 0) {
				if (!strcmp(driver,"alsa") || !strcmp(driver,"oss")) {
					if (param && strcmp(param,"default")) {
						ao_append_option(&self->ao_options,"dev",param);
					}
				}
				else if (!strcmp(driver,"nas")) {
					if (!param) {
						PyErr_SetString(PyExc_RuntimeError,"NAS server not set");
						return -1;
					}
					ao_append_option(&self->ao_options,"server",param);
				}
				else if (!strcmp(driver,"pulse")) {
					if (param && strcmp(param,"default")) {
						ao_append_option(&self->ao_options,"server",param);
					}
				}
			}
		}
		else self->ao_driver = ao_default_driver_id();
		if (self->ao_driver == -1) {
			PyErr_SetString(PyExc_RuntimeError,"Cannot find libao driver");
			return -1;
		}
	}
#endif
#ifdef HAVE_PULSE
	else {
		self->p_simple=NULL;
		self->pulse_use_drain=0;
		self->pulse_margin=300;
		if (param && strcmp(param,"default")) self->p_server=strdup(param);
		self->p_name=strdup(name?name:"speech-dispatcher");
	}
#endif
        
	return 0;
}

#if 0
static int init_ao(pyspda_PySpdaObject *self,int rate,int channels)
{
	self->ao_format.bits = 16;
	self->ao_format.channels = channels;
	self->ao_format.rate = rate;
	self->ao_format.byte_format = AO_FMT_LITTLE;
	self->ao_device = ao_open_live(self->ao_driver, &self->ao_format, self->ao_options);
	self->samples_sent=0;
	if (!self->ao_device) {
		return -1;
	}
	return 0;
}
#endif

static int init_alsa(pyspda_PySpdaObject *self,int rate,int channels)
{
	snd_pcm_hw_params_t *params;
	self->alsa_channels=channels;
	self->alsa_rate=rate;
	if (snd_pcm_open(&self->alsa_pcm_handle,
		self->alsa_device_name,
		SND_PCM_STREAM_PLAYBACK,
		0)) return -1;
	snd_pcm_hw_params_alloca(&params);
	snd_pcm_hw_params_any(self->alsa_pcm_handle, params);
 
/* Set parameters */
	if (snd_pcm_hw_params_set_access(self->alsa_pcm_handle, params,
		SND_PCM_ACCESS_RW_INTERLEAVED) < 0) goto bad_param;
	if (snd_pcm_hw_params_set_format(self->alsa_pcm_handle, params,
		SND_PCM_FORMAT_S16_LE) < 0) goto bad_param;
	if (snd_pcm_hw_params_set_channels(self->alsa_pcm_handle, params, self->alsa_channels) < 0) {
		goto bad_param;
	}
	if(snd_pcm_hw_params_set_rate(self->alsa_pcm_handle, params, self->alsa_rate, 0) < 0) {
		goto bad_param;
	}
	/* Write parameters */
	if (snd_pcm_hw_params(self->alsa_pcm_handle, params) < 0) {
		goto bad_param;
	}
	snd_pcm_hw_params_get_period_size(params, &self->alsa_frames, 0);
	int bs=self->alsa_frames * self->alsa_channels * 2 /* 2 -> sample size */;
	if (bs != self->alsa_buff_size) {
		if (self->alsa_buffer) free(self->alsa_buffer);
		self->alsa_buff_size=bs;
		self->alsa_buffer = (char *) malloc(self->alsa_buff_size); 
		
	}
	return 0;
 
bad_param:
	snd_pcm_close(self->alsa_pcm_handle);
	self->alsa_pcm_handle=NULL;
	return -1;
}


static void revolume(short *wave,int len,double volume)
{
	int n;
	while (len-- > 0) {
		n=(*wave) * volume;
		*wave++ = n;
	}
}

static void apply_contrast(signed short *data,int len,double contrast_level)
{
	while (len-- > 0) {
		double d=(*data) * (M_PI_2/32768.0);
		*data++=sin(d + contrast_level * sin(d * 4)) * 32767;
	}
}

#define DEFAULT_MIN_AUDLEN 100
#ifdef HAVE_PULSE
static int _pulse_open(pyspda_PySpdaObject *self,int rate,int channels)
{
	pa_buffer_attr buffAttr;
	pa_sample_spec ss;
	int error;

	ss.rate = rate;
	ss.channels = channels;
	ss.format = PA_SAMPLE_S16LE;

	buffAttr.maxlength = (uint32_t)-1;
	buffAttr.tlength = DEFAULT_MIN_AUDLEN;
	buffAttr.prebuf = (uint32_t)-1;
	buffAttr.minreq = (uint32_t)-1;
	buffAttr.fragsize = (uint32_t)-1;
	self->p_simple = pa_simple_new(
				self->p_server,
				self->p_name,
				PA_STREAM_PLAYBACK,
				     NULL, "playback", &ss, NULL, &buffAttr, &error);
	if (!self->p_simple) {
		PyErr_SetString(PyExc_RuntimeError,pa_strerror(error));
		return -1;
	}
	self->p_current_rate = rate;
	self->p_channels=channels;
	self->samples_sent=0;
	return 0;
}

static char zeroes[512];

static void play_pulse_margin(pyspda_PySpdaObject *self)
{
	int zs=(self->p_channels*self->pulse_margin * self->p_current_rate)/500;
	Py_BEGIN_ALLOW_THREADS
	while (zs > 0) {
		if (self->stop_playing) break;
		int n=zs;
		if (n>512) n=512;
		if(pa_simple_write(self->p_simple, zeroes, n, NULL) < 0) {
			pa_simple_free(self->p_simple);
			self->p_simple=NULL;
			break;
		}
		self->samples_sent+=n/(2*self->p_channels);
		zs-=n;
	}
	Py_END_ALLOW_THREADS
}

static void pulse_finalize(pyspda_PySpdaObject *self)
{
	if (self->p_simple) {
		if (self->pulse_use_drain) {
			pa_simple_drain(self->p_simple,NULL);
		}
		else {
			play_pulse_margin(self);
		}
		if (self->p_simple) {
			pa_simple_free(self->p_simple);
			self->p_simple=0;
		}
	}
}
#endif

static void alsa_flush(pyspda_PySpdaObject *self)
{
	if (self->alsa_buffer_pos) {
		memset(self->alsa_buffer+self->alsa_buffer_pos,
			0,
			self->alsa_buff_size-self->alsa_buffer_pos);
		snd_pcm_writei(self->alsa_pcm_handle,
			self->alsa_buffer,
			self->alsa_frames);
		self->alsa_buffer_pos=0;
	}
}

static int init_audio(pyspda_PySpdaObject *self,int rate, int channels)
{
        if (self->audio_mode == PSD_AMODE_ALSA) {
		if (self->alsa_playing && 
			(self->alsa_rate != rate  || self->alsa_channels != channels)) {
				alsa_flush(self);
				snd_pcm_drain(self->alsa_pcm_handle);
				snd_pcm_close(self->alsa_pcm_handle);
				self->alsa_pcm_handle=NULL;
				self->alsa_playing=0;
		}
		if (!self->alsa_playing) {
			if (init_alsa(self,rate,channels) < 0) {
				return -1;
			}
			self->alsa_playing=1;
		}
		return 0;
		
	}
#if 0
	if (self->audio_mode == PSD_AMODE_AO) {
		if (self->ao_playing &&
				(self->ao_format.rate != rate  || self->ao_format.channels != channels) ){
			ao_close(self->ao_device);
			self->ao_device=0;
			self->ao_playing=0;
		}
		if (!self->ao_playing) {
			if (init_ao(self,rate,channels) < 0) return -1;
			self->ao_playing=1;
		}
		return 0;
	}
#endif
#ifdef HAVE_PULSE
	if (self->audio_mode == PSD_AMODE_PULSE) {
		if (self->p_simple) {
			if (self->p_current_rate == rate && self->p_channels == channels) {
				return 0;
			}
			pulse_finalize(self);
		}
		if (_pulse_open(self,rate,channels)<0) {
			return -1;
		}
		return 0;
	}
#endif
	return -1;
}

static void PySpda_dealloc(pyspda_PySpdaObject *self)
{
    if (self->audio_mode == PSD_AMODE_ALSA) {
	    if (self->alsa_pcm_handle) {
		    snd_pcm_close(self->alsa_pcm_handle);
		    self->alsa_pcm_handle = NULL;
		}
		if (self->alsa_buffer) {
			free(self->alsa_buffer);
			self->alsa_buffer=0;
		}
	}
#if 0
    if (self->audio_mode == PSD_AMODE_AO) {
		if (self->ao_device) {
			ao_close(self->ao_device);
			self->ao_device=NULL;
			if (self->ao_options) ao_free_options(self->ao_options);
		}
	}
#endif
#ifdef HAVE_PULSE
	if (self->audio_mode == PSD_AMODE_PULSE) {
		if (self->p_simple) {
			pa_simple_free(self->p_simple);
			self->p_simple=NULL;
		}
	}
	if (self->p_server) free(self->p_server);
	if (self->p_name) free(self->p_name);
#endif
    self->ob_type->tp_free((PyObject*)self);
}

static int play_alsa(pyspda_PySpdaObject *self,char *wave, int bytecount, int channels)
{
	static int q=0;
	while (bytecount > 0) {
		int n=self->alsa_buff_size-self->alsa_buffer_pos;
		if (bytecount < n) n=bytecount;
		memcpy(self->alsa_buffer+self->alsa_buffer_pos, wave, n);
		self->alsa_buffer_pos += n;
		wave += n;
		bytecount -= n;
		q += n;
		if (self->alsa_buffer_pos < self->alsa_buff_size) {
			break;
		}
		int pcm=snd_pcm_writei(self->alsa_pcm_handle,
			self->alsa_buffer,
			self->alsa_frames);
		self->alsa_buffer_pos=0;
		if (pcm == -EPIPE) {
			snd_pcm_prepare(self->alsa_pcm_handle);
		}
		else if (pcm < 0) {
			snd_pcm_close(self->alsa_pcm_handle);
			self->alsa_playing=0;
			self->alsa_pcm_handle=NULL;
			return -1;
		}
	}
	return 0;
}		

static int play_small_part(pyspda_PySpdaObject *self,char *wave,int bytecount,int channels)
{
	
        if (self->audio_mode == PSD_AMODE_ALSA) {
		if (play_alsa(self,wave,bytecount,channels) < 0) {
			return -1;
		}
		return 0;
	}
#if 0
    else if (self->audio_mode == PSD_AMODE_AO) {
	if (!ao_play(self->ao_device,wave,bytecount)) {
	    ao_close(self->ao_device);
	    self->ao_playing=0;
	    return -1;
	}
    }
#endif
#ifdef HAVE_PULSE
    else if (self->audio_mode == PSD_AMODE_PULSE) {
	if(pa_simple_write(self->p_simple, wave, bytecount, NULL) < 0) {
	    pa_simple_free(self->p_simple);
	    self->p_simple=NULL;
	    return -1;
	}
    }
#endif
    else {
	return -1;
    }
    self->samples_sent+=bytecount/(2*channels);
    return 0;    
}

static int play_part(pyspda_PySpdaObject *self,char *wave,int count,int rate,int channels)
{
    //printf("Playing %d bytes\n",count);
    if (init_audio(self,rate,channels) < 0) {
		PyErr_SetString(PyExc_RuntimeError,"Audio not initialized");
		return -1;
	}
	Py_BEGIN_ALLOW_THREADS
	catch_signal();
	while (count > 0) {
		int n;
		if (_signal_got || self->stop_playing) {
			count=0;
			break;
		}
		
		n=self->fragment_size*2*channels;
		if (n > count) n=count;
                //printf("Playing part %d bytes\n",n);
		if (play_small_part(self,wave,n,channels)) {
		    break;
		}
		count-=n;
		wave+=n;
	}
	uncatch_signal();
	Py_END_ALLOW_THREADS
	return 0;
}

static void do_equalize(pyspda_PySpdaObject *self,
        short *wave,
        int count,
        int samplerate,
        int channels)
{
        int i;
        if (!self->eqenabled) return;

        if (samplerate != self->e_samplerate ||
                channels != self->e_channels) {
                        self->e_channels=channels;
                        self->e_samplerate=samplerate;
                        for (i=0;i<2;i++) {
                                init_equaliser(&self->es[i],800,2400,samplerate);
                        }
        }
        for (i=0;i<2;i++) {
                self->es[i].lg=self->elg;
                self->es[i].mg=self->emg;
                self->es[i].hg=self->ehg;
        }
        count /= 2;
        for (i=0;i<count; i++) {
                double d;
                self -> tc= (self->tc + 1) % self->e_channels;
                d = equalise(&self->es[self->tc],((double)wave[i])/(32768.0 * self->edivisor));
                
                if (d<-1.0) d=-1.0;
                else if (d > 1.0) d=1.0;
                wave[i]=32767 *d;
        }
}
        

static int play_part_s(pyspda_PySpdaObject *self,
			char *wave,
			int count,
			int samplerate,
			int channels,
			int pitch,
			int rate)
{
    sonicStream sonic;
    int part_count;
    int n;
    short sonic_buffer[8192];
    
    pitch=ensure(pitch,-100,100);
    rate=ensure(rate,-50,100);
    
    do_equalize(self,(short *)wave,count,samplerate,channels);
    
    if (!pitch && !rate) {
	return play_part(self,wave,count,samplerate,channels);
    }
    
    if (init_audio(self,samplerate,channels) < 0) {
        PyErr_SetString(PyExc_RuntimeError,"Audio not initialized");
        return -1;
    }

    sonic=sonicCreateStream(samplerate,1);
    if (rate) {
	sonicSetSpeed(sonic,(100.0+(double)rate)/100.0);
    }
    if (pitch) {
	sonicSetPitch(sonic,(100.0+(double)((pitch<0)?pitch/2:pitch))/100.0);
    }
    Py_BEGIN_ALLOW_THREADS
    catch_signal();
    while (count > 0) {
        if (_signal_got || self->stop_playing) {
            count=0;
            break;
        }
	part_count = 32768;
	if (part_count > count) {
	    part_count=count;
	}
	sonicWriteShortToStream(sonic,(short *)wave,part_count/2);
	wave += part_count;
	count -= part_count;
	for (;;) {
            if (_signal_got || self->stop_playing) {
                count=0;
                break;
            }
	    n=sonicReadShortFromStream(sonic,sonic_buffer,self->fragment_size * channels);
	    if (n <= 0) break;
	    if (play_small_part(self,(char *)sonic_buffer,2*n,channels)) {
		count=0;
		break;
	    }
	} // end of sonic loop
	    
    } // end of wave loop
    sonicFlushStream(sonic);
    for (;;) {
        if (_signal_got || self->stop_playing) {
            break;
        }
        n=sonicReadShortFromStream(sonic,sonic_buffer,256);
        if (n <= 0) break;
	if (play_small_part(self,(char *)sonic_buffer,2*n,channels)) {
	    break;
        }
    }
    uncatch_signal();
    Py_END_ALLOW_THREADS
    sonicDestroyStream(sonic);
    return 0;
}
    

static PyObject *PySpda_play(pyspda_PySpdaObject *self,
			PyObject * args, PyObject *kwds)
{
	static char *kwlist[]={"wave","count","freq","channels","pitch","rate",NULL};
	int samplerate=0;
	int count=-1;
	int channels=1;
	char *wave;
	int wavelen;
	int pitch=0;
	int rate=0;
	if (self->stop_playing) {
		Py_RETURN_NONE;
	}
	if (! PyArg_ParseTupleAndKeywords(args, kwds, "s#|iiiii", kwlist,
		&wave,
		&wavelen,
		&count,
		&samplerate,
		&channels,
		&pitch,
		&rate)) return NULL;
	if (channels < 1 || channels > 2) {
		PyErr_SetString(PyExc_RuntimeError,"Bad channel number");
	}
	if (!samplerate) {
		if (self->audio_mode == PSD_AMODE_ALSA) {
			if (!self->alsa_playing) {
				PyErr_SetString(PyExc_RuntimeError,"Samplerate not set");
				return NULL;
			}
			samplerate=self->alsa_rate;
		}
#if 0
		if (self->audio_mode == PSD_AMODE_AO) {
			if (!self->ao_playing) {
				PyErr_SetString(PyExc_RuntimeError,"Samplerate not set");
				return NULL;
			}
			samplerate=self->ao_format.rate;
		}
#endif
#ifdef HAVE_PULSE
		else if (self->audio_mode == PSD_AMODE_PULSE) {
			if (!self->p_simple) {
				PyErr_SetString(PyExc_RuntimeError,"Samplerate not set");
				return NULL;
			}
			samplerate=self->p_current_rate;
		}
#endif
	}
    if (count < 0) count=wavelen;
    if (self->volume < 100) {
		revolume((short *)wave,count/2,self->d_volume);
	}
	if (self->contrast > 0) {
		apply_contrast((short *)wave,count/2,self->d_contrast);
	}
    if (play_part_s(self,wave,count,samplerate,channels,pitch,rate)<0) {
		return NULL;
	}
    if (_signal_got) {
	PyErr_SetString(PyExc_KeyboardInterrupt,"Audio play interrupted");
	return NULL;
    }
    Py_RETURN_NONE;
    
}


static void audio_end(pyspda_PySpdaObject *self)
{
	if (self->audio_mode == PSD_AMODE_ALSA) {
		if (self->alsa_playing) {
			alsa_flush(self);
			snd_pcm_drain(self->alsa_pcm_handle);
			snd_pcm_close(self->alsa_pcm_handle);
			self->alsa_pcm_handle=NULL;
			self->alsa_playing=0;
		}
	}
#if 0
	if (self->audio_mode == PSD_AMODE_AO) {
		if (self->ao_playing) {
			ao_close(self->ao_device);
			self->ao_device=NULL;
			self->ao_playing=0;
		}
	}
#endif
#ifdef HAVE_PULSE
	else if (self->audio_mode == PSD_AMODE_PULSE) {
		if (self->p_simple) {
			if (self->pulse_use_drain) {
				pa_simple_drain(self->p_simple,NULL);
				pa_simple_free(self->p_simple);
				self->p_simple=0;
			}
		}
	}
#endif
}
static PyObject *PySpda_end(pyspda_PySpdaObject *self,
			PyObject * args, PyObject *kwds)
{
	audio_end(self);
	Py_RETURN_NONE;
}


static PyObject *PySpda_enable(pyspda_PySpdaObject *self,
			PyObject * args, PyObject *kwds)
{
	self->stop_playing=0;
	Py_RETURN_NONE;
}

static PyObject *PySpda_stop(pyspda_PySpdaObject *self,
			PyObject * args, PyObject *kwds)
{
	self->stop_playing=1;
	Py_RETURN_NONE;
}

static PyObject *PySpda_finish(pyspda_PySpdaObject *self,
			PyObject * args, PyObject *kwds)
{
#ifdef HAVE_PULSE
	if (self->audio_mode == PSD_AMODE_PULSE) {
		if (self->p_simple) {
			pulse_finalize(self);
		}
	}
#endif
	Py_RETURN_NONE;
}

static PyObject *PySpda_beep(pyspda_PySpdaObject *self,
			PyObject * args, PyObject *kwds)
{
    short buffer[16384];
    int i,mode=0;
    double p,step;

    if (!PyArg_ParseTuple(args, "|i", &mode)) return NULL;
    
    for (i=0,p=0;i<16384;i++) {
	buffer[i]=sin(p) * 30000;
	if (mode > 0) step=3.141592 * ((i+16384)/16384.0) / 32.0;
	else if (mode < 0) step=3.141592 * ((32768 - i)/16384.0) /32.0;
	else step= 3.141592 / 32.0;
	p += step;
    }
    play_part(self,(char *)buffer,2*16386,16000,1);
    audio_end(self);
    Py_RETURN_NONE;
}

static PyObject *PySpda_playFile(pyspda_PySpdaObject *self,
			PyObject * args, PyObject *kwds)
{
	SF_INFO info;
	SNDFILE *f;
	int nsamp;
	char *fname=NULL;
	char buffer[8192];
    if (!PyArg_ParseTuple(args, "s", &fname)) return NULL;

	f=sf_open(fname,SFM_READ,&info);
	if (!f) {
		Py_RETURN_NONE;
	}
	if (info.channels < 1 || info.channels > 2) {
		sf_close(f);
		Py_RETURN_NONE;
	}
	nsamp=info.frames*info.channels;
	while (nsamp > 0) {
		int n=4096;
		if (n>nsamp) n=nsamp;
		n=sf_read_short(f,(short *)buffer,n);
		if (n <= 0) break;
		nsamp -= n;
		play_part(self,(char *)buffer,2*n,info.samplerate,info.channels);
	}
	sf_close(f);
	audio_end(self);
	Py_RETURN_NONE;
}

static PyObject *PySpda_vol(pyspda_PySpdaObject *self,
			PyObject * args, PyObject *kwds)
{
	int vol;
	if (!PyArg_ParseTuple(args, "i", &vol)) return NULL;
	self->volume=ensure(vol,-100,100);
	self->d_volume=(self->volume+100.0)/200.0;
	Py_RETURN_NONE;
}

static PyObject *PySpda_contrast(pyspda_PySpdaObject *self,
			PyObject * args, PyObject *kwds)
{
	int con;
	if (!PyArg_ParseTuple(args, "i", &con)) return NULL;
	self->contrast=ensure(con,0,100);
	self->d_contrast=0.03+self->contrast/1000.0;
	Py_RETURN_NONE;
}

static PyObject *PySpda_equalizer(pyspda_PySpdaObject *self,
			PyObject * args, PyObject *kwds)
{
        double flow,fmid,fhig;
        int is_one(double t)
        {
                return (t >= 0.99 && t <=1.01);
        }
        if (!PyArg_ParseTuple(args,"dd|d",&flow,&fmid,&fhig)) return NULL;
        self->edivisor=self->elg=flow;
        self->emg=fmid;if (fmid > self->edivisor) self->edivisor=fmid;
        self->ehg=fhig;if (fhig > self->edivisor) self->edivisor=fhig;
        self->eqenabled=!(is_one(flow) && is_one(fhig) && is_one(fmid));
        Py_RETURN_NONE;
}


static PyObject *PySpda_setMargin(pyspda_PySpdaObject *self,
			PyObject * args, PyObject *kwds)
{
#ifdef HAVE_PULSE
	int mrg;
	if (!PyArg_ParseTuple(args, "i", &mrg)) return NULL;
	self->pulse_margin=ensure(mrg,0,1000);
#endif
	Py_RETURN_NONE;
}

static PyObject *PySpda_setFragment(pyspda_PySpdaObject *self,
			PyObject * args, PyObject *kwds)
{
	int m;
	if (!PyArg_ParseTuple(args, "i", &m)) return NULL;
	self->fragment_size=ensure(m,128,4096);
	Py_RETURN_NONE;
}

static PyObject *PySpda_allowInterrupt(pyspda_PySpdaObject *self,
			PyObject * args, PyObject *kwds)
{
    int m;
    if (!PyArg_ParseTuple(args, "i", &m)) return NULL;
    _allow_signal_catch=m?1:0;
    Py_RETURN_NONE;
}

static PyMethodDef PySpda_ModMethods[]={
	{"allowInterrupt",(PyCFunction)PySpda_allowInterrupt,METH_VARARGS,
		"allowInterrupt(enable)\nEnable break play by ctrl-c\nUse only in single-thread applications\n"},
	{NULL,NULL,0,NULL}
};

static PyMethodDef PySpda_methods[]={
	{"enable",(PyCFunction)PySpda_enable,METH_VARARGS,
		"enable()\nClear stop flag in object"},
	{"vol",(PyCFunction)PySpda_vol,METH_VARARGS,
		"vol(v)\nSet volume (range -100 to 100)"},
	{"contrast",(PyCFunction)PySpda_contrast,METH_VARARGS,
		"contrast(c)\nSet contrast (range 0 to 100)"},
	{"equalizer",(PyCFunction)PySpda_equalizer,METH_VARARGS,
		"equalizer(lo,md,hi)\nSet equalizer (double, double,double)"},
	{"setMargin",(PyCFunction)PySpda_setMargin,METH_VARARGS,
		"setMargin(msec)\nSet pulse_audio silence margin (range 0 to 1000)"},
	{"setFragment",(PyCFunction)PySpda_setFragment,METH_VARARGS,
		"setFragment(samples)\nSet audio fragment size (range 128 to 4096)"},
	{"stop",(PyCFunction)PySpda_stop,METH_VARARGS,
		"stop()\nSet stop flag in object"},
	{"play",(PyCFunction)PySpda_play,METH_VARARGS | METH_KEYWORDS,
		"play(wave,count=None,freq=None,channels=1,pitch=0,rate=0)\n	Play wave."},
	{"playFile",(PyCFunction)PySpda_playFile,METH_VARARGS,
		"playFile(filename)\n	Play file\n"},
	{"beep",(PyCFunction)PySpda_beep,METH_VARARGS,
		"beep()\nMake beep"},
	{"allowInterrupt",(PyCFunction)PySpda_allowInterrupt,METH_VARARGS,
		"allowInterrupt(enable)\nEnable break play by ctrl-c\nUse only in single-thread applications\n"},
	{"end",(PyCFunction)PySpda_end,METH_VARARGS,
		"end()\nMust be called when full wave was played"},
	{"finish",(PyCFunction)PySpda_finish,METH_VARARGS,
		"finish()\nMust be called before break"},
	{NULL,NULL,0,NULL}
};


static PyObject *PySpda_getPM(pyspda_PySpdaObject *obj,void *closure)
{
    return PyLong_FromLong(
#ifdef HAVE_PULSE				     
				     obj->pulse_margin
#else
0
#endif
				     );
}

static int PySpda_setPM(pyspda_PySpdaObject *obj,PyObject *value,void *closure)
{
    if (value == NULL) {
        PyErr_SetString(PyExc_TypeError, "Cannot delete the attribute");
        return -1;
    }
    if (!PyInt_Check(value)) {
	PyErr_SetString(PyExc_TypeError, "Numeric argument expected");
        return -1;
    }	
    obj->pulse_margin=ensure(PyInt_AsLong(value),0,1000);
    return 0;
}

static PyObject *PySpda_getFragment(pyspda_PySpdaObject *obj,void *closure)
{
    return PyLong_FromLong(obj->fragment_size);
}

static int PySpda_setFragmentX(pyspda_PySpdaObject *obj,PyObject *value,void *closure)
{
    if (value == NULL) {
        PyErr_SetString(PyExc_TypeError, "Cannot delete the attribute");
        return -1;
    }
    if (!PyInt_Check(value)) {
	PyErr_SetString(PyExc_TypeError, "Numeric argument expected");
        return -1;
    }	
    obj->fragment_size=ensure(PyInt_AsLong(value),128,4096);
    return 0;
}

static PyObject *PySpda_getStopped(pyspda_PySpdaObject *obj,void *closure)
{
    return PyLong_FromLong(obj->stop_playing);
}

static PyObject *PySpda_getSampSent(pyspda_PySpdaObject *obj,void *closure)
{
    return PyLong_FromLong(obj->samples_sent);
}
    

static PyGetSetDef PySpda_getseters[] = {
    {"pulse_margin",
     (getter)PySpda_getPM, (setter)PySpda_setPM,
     "Pulse audio margin in liliseconds",
     NULL},
    {"fragment",
     (getter)PySpda_getFragment, (setter)PySpda_setFragmentX,
     "Audio fragment size",
     NULL},
    {"stopped",
     (getter)PySpda_getStopped, NULL,
     "Audio stopped (readonly)",
     NULL},
    {"samples_sent",
     (getter)PySpda_getSampSent, NULL,
     "Audio samples sent (readonly)",
     NULL},
    {NULL}  /* Sentinel */
};
static PyTypeObject pyspda_PySpdaType = {
#if PY_MAJOR_VERSION == 3
    PyVarObject_HEAD_INIT(NULL, 0)
#else
    PyObject_HEAD_INIT(NULL)
    0,                         /*ob_size*/
#endif
    "pyspda.Spda",             /*tp_name*/
    sizeof(pyspda_PySpdaObject), /*tp_basicsize*/
    0,                         /*tp_itemsize*/
    (destructor)PySpda_dealloc,                         /*tp_dealloc*/
    0,                         /*tp_print*/
    0,				/*tp_getattr*/
    0,                         /*tp_setattr*/
    0,                         /*tp_compare*/
    0,                         /*tp_repr*/
    0,                         /*tp_as_number*/
    0,                         /*tp_as_sequence*/
    0,                         /*tp_as_mapping*/
    0,                         /*tp_hash */
    0,                         /*tp_call*/
    0,                         /*tp_str*/
    0,                         /*tp_getattro*/
    0,                         /*tp_setattro*/
    0,                         /*tp_as_buffer*/
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,        /*tp_flags*/
    "Audio object:\nSpda(type,driver='default',name='',param='default)\n\
	where type is one of AUDIO_AO or AUDIO_PULSE",           /* tp_doc */
    0,		               /* tp_traverse */
    0,		               /* tp_clear */
    0,		               /* tp_richcompare */
    0,		               /* tp_weaklistoffset */
    0,		               /* tp_iter */
    0,		               /* tp_iternext */
    PySpda_methods,         /* tp_methods */
    0,                         /* tp_members */
    PySpda_getseters,                         /* tp_getset */
    0,                         /* tp_base */
    0,                         /* tp_dict */
    0,                         /* tp_descr_get */
    0,                         /* tp_descr_set */
    0,                         /* tp_dictoffset */
    (initproc)PySpda_init,  /* tp_init */
};

#ifndef PyMODINIT_FUNC	/* declarations for DLL import/export */
#define PyMODINIT_FUNC void
#endif
PyMODINIT_FUNC
initpyspda(void)
{
    PyObject* m; int i;

    pyspda_PySpdaType.tp_new = PyType_GenericNew;
    if (PyType_Ready(&pyspda_PySpdaType) < 0)
        return;

    m = Py_InitModule3("pyspda", PySpda_ModMethods,
                       "Spda - Speech-Displatcher audio output");
    Py_INCREF(&pyspda_PySpdaType);
    PyModule_AddObject(m, "Spda", (PyObject *)&pyspda_PySpdaType);


    for (i=0;pyspda_consts[i].name;i++) {
        PyModule_AddObject(m, pyspda_consts[i].name, Py_BuildValue("i",pyspda_consts[i].value));
    }
}
