#!/usr/bin/env python
#coding: utf-8

# gui.py - GTK3 GUI for SAPI Book Reader
# Copyright (C) Bohdan R. Rau 2013 <ethanak@polip.com>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this program.  If not, write to:
# 	The Free Software Foundation, Inc.,
# 	51 Franklin Street, Fifth Floor
# 	Boston, MA  02110-1301, USA.


from gi.repository import Gtk,GtkSource,Pango,GLib,GObject,Gdk
import re,os,sys

class gui(Gtk.Window):
    @staticmethod
    def exit(*args):
        try:
            pass
        except:
            pass
        from ctypes import CDLL
        CDLL("libc.so.6").exit(0)

    def place_cursor(self, pos):
        self._cursor=pos
        self.last_sentence=None
        GLib.timeout_add(300,self._init_cursor,None)

    def set_text(self, text,pos=0):
        self.buffer.set_text(text)
        self._cursor=pos
        lgs=self.core.language(text)
        if lgs:
            lang=self.voices[self.voicecombo.get_active()][1]
            fd=False
            vname=self.core.get_params([lang+'_voice'])[lang+'_voice']
            vspeed=self.core.get_params([lang+'_speed'])[lang+'_speed']
            if vspeed:
                self.voicespeed.set_value(vspeed)
            if vname:
                for i,a in enumerate(self.voices):
                    if a[1]==lgs[0] and a[0] == vname:
                        self.voicecombo.set_active(i)
                        fd=True
                        break
            if lang == lgs[0]:
                fd=True
            if not fd:
                for i,a in enumerate(self.voices):
                    if a[1] == lgs[0]:
                        self.voicecombo.set_active(i)
                        fd=True
                        break
            if not fd:
                self.alert('Unsupported language: '+lgs[1])
            else:
                self.llang_f=lgs[0]
        GLib.timeout_add(300,self._init_cursor,None)
        
    def alert(self,txt,iserror=False):
        dialog=Gtk.MessageDialog(self,3,Gtk.MessageType.ERROR if iserror else Gtk.MessageType.INFO,Gtk.ButtonsType.OK,txt)
        dialog.run()
        dialog.destroy()
    
    def ask(self,txt):
        dialog=Gtk.MessageDialog(self,3,Gtk.MessageType.QUESTION,Gtk.ButtonsType.YES_NO,txt)
        rc=dialog.run()
        dialog.destroy()
        return rc==Gtk.ResponseType.YES
    
    def loop(self):
        Gtk.main()
    
    def set_play_position(self, pos,count):
        self.position=(pos,count)
        
    def end_of_book(self):
        self.speaking=False
        
    def set_editable(self,editable):
        self.editable=editable
        self.edit_button.set_active(editable)
        
    
    def query(self,what,title='Question'):
        ls=[]
        dialog=Gtk.Dialog(title,self,3)
        dialog.add_button(Gtk.STOCK_CANCEL,Gtk.ResponseType.Cancel)
        dialog.add_button(Gtk.STOCK_OK,Gtk.ResponseType.OK)
        inn=dialog.get_content_area()
        grid=Gtk.Grid()
        inn.pack_start(grid,0,0,2)
        for i,a in enumerate(what):
            if isinstance(a,str) or isinstance(a,unicode):
                label=a
                key=a
            else:
                label=a[0]
                key=a[1]
            lb=Gtk.Label(label)
            grid.attach(lb,0,i,1,1)
            entry=Gtk.Entry()
            entry.set_size_request(250,-1)
            grid.attach(entry,1,i,1,1)
            ls.append((key,entry))
        inn.show_all()
        r=dialog.run()
        rc=None
        if r == Gtk.ResponseType.OK:
            rc={}
            for a in ls:
                rc[a[0]]=a[1].get_text()
        dialog.destroy()
        return rc

    def new_file_started(self,bookname):
        self.buffer.set_modified(False)
        self.edit_button.set_active(False)
        self.editable=False
        self.core.enable_bookmarks()
        self._show_bookmarks()
        self._last_dir=os.path.dirname(bookname)
        self._last_file=bookname
        self.set_title(os.path.basename(bookname))
            

        
    
    # private
    
    
    def _find_dialog(self,*args):
        if self.speaking:
            return
        d=self._create_find_dialog(False)
        while True:
            rc=d.run()
            if rc != Gtk.ResponseType.OK:
                d.destroy()
                return
            txt=d.find_entry.get_text()
            if txt:
                break
        rs={
            'find':txt,
            'replace':'',
            're':d.cb_regexp.get_active(),
            'cs':d.cb_sensitive.get_active(),
            'sw':d.cb_startword.get_active(),
            'ew':d.cb_endword.get_active(),
            'iw':d.cb_isword.get_active()
        }
        if len(self.findlist) == 0 or self.findlist[-1] != rs:
            self.findlist.append(rs)
        fwd = not d.cb_backward.get_active()
        d.destroy()
        
        self._local_find(fwd,rs)
    
    def _find_next(self,*args):
        if len(self.findlist) == 0:
            return self._find_dialog()
        rs=self.findlist[-1]
        return self._local_find(True,rs)

    def _find_previous(self,*args):
        if len(self.findlist) == 0:
            return self._find_dialog()
        rs=self.findlist[-1]
        return self._local_find(False,rs)
    
    def _find_by_regexp(self, fwd, rs):
        flags=re.UNICODE
        if not rs['cs']:
            flags |= re.IGNORECASE
        try:
            r=re.compile(rs['find'],flags)
        except:
            import sys
            self.alert(sys.exc_info()[1].message,True)
            return
        
        bounds=self.buffer.get_selection_bounds()
        if bounds:
            itr1=bounds[1 if fwd else 0]
        else:
            tm=self.buffer.get_insert()
            itr1=self.buffer.get_iter_at_mark(tm)
        if fwd:
            itr2=self.buffer.get_end_iter()
            body=self.buffer.get_text(itr1,itr2,False)
            if not isinstance(body,unicode):
                body=body.decode('utf-8')
            m=r.search(body)
            if not m:
                self.alert('Not found',True)
                return
            itr2=itr1.copy()
            itr1.forward_chars(m.start())
            itr2.forward_chars(m.end())
        else:
            itr2=self.buffer.get_start_iter()
            body=self.buffer.get_text(itr2,itr1,False)
            if not isinstance(body,unicode):
                body=body.decode('utf-8')
            m=None
            for m in r.finditer(body):
                pass
            if not m:
                self.alert('Not found',True)
                return
            itr1=self.buffer.get_iter_at_offset(m.start())
            itr2=self.buffer.get_iter_at_offset(m.end())
        self.buffer.select_range(itr1,itr2)
        self.view.scroll_to_iter(itr1,0.05,False,0,0)
            
            
            
        
        
    
    
    def _local_find(self,fwd,rs):
        if rs['re']:
            return self._find_by_regexp(fwd,rs)
        bounds=self.buffer.get_selection_bounds()
        if bounds:
            itr=bounds[1 if fwd else 0]
        else:
            tm=self.buffer.get_insert()
            itr=self.buffer.get_iter_at_mark(tm)
        m=Gtk.TextSearchFlags.VISIBLE_ONLY | Gtk.TextSearchFlags.TEXT_ONLY
        if not rs['cs']:
            m |= Gtk.TextSearchFlags.CASE_INSENSITIVE
        while True:
            if fwd:
                ires=itr.forward_search(rs['find'],m,None)
            else:
                ires=itr.backward_search(rs['find'],m,None)
            if not ires:
                self.alert('Not found')
                return
            if rs['sw'] or rs['iw']:
                if not ires[0].starts_word():
                    if fwd:
                        itr=ires[1]
                    else:
                        itr=ires[0]
                    continue
            if rs['ew'] or rs['iw']:
                if not ires[1].ends_word():
                    if fwd:
                        itr=ires[1]
                    else:
                        itr=ires[0]
                    continue
            self.buffer.select_range(ires[0],ires[1])
            self.view.scroll_to_iter(ires[0],0.05,False,0,0)
            return
                    

            

    def _rword(self,button,d):
        if not button.get_active():
            return
        a=d[0]
        d=d[1]
        if a != 0:
            d.cb_regexp.set_active(False)
        if a != 1:
            d.cb_startword.set_active(False)
        if a != 2:
            d.cb_endword.set_active(False)
        if a != 3:
            d.cb_isword.set_active(False)
    
    
    def _search_history(self,dialog,direction):
        if len(self.findlist) == 0:
            return
        if dialog.history_number == 0:
            if not direction:
                return
            else:
                rs={
                    'find':dialog.find_entry.get_text(),
                    'replace':dialog.replace_entry.get_text if dialog.is_replace_dialog else '',
                    're':dialog.cb_regexp.get_active(),
                    'cs':dialog.cb_sensitive.get_active(),
                    'sw':dialog.cb_startword.get_active(),
                    'ew':dialog.cb_endword.get_active(),
                    'iw':dialog.cb_isword.get_active()
                }
                dialog.manual_form=rs
        if direction:
            if dialog.history_number >= len(self.findlist):
                return
            else:
                dialog.history_number += 1
        else:
            dialog.history_number -= 1
        if dialog.history_number == 0:
            rs=dialog.manual_form
        else:
            rs=self.findlist[-dialog.history_number]
        dialog.find_entry.set_text(rs['find'])
        if dialog.is_replace_dialog:
            dialog.replace_entry.set_text(rs['replace'])
        dialog.cb_regexp.set_active(rs['re'])
        dialog.cb_sensitive.set_active(rs['cs'])
        dialog.cb_startword.set_active(rs['sw'])
        dialog.cb_endword.set_active(rs['ew'])
        dialog.cb_isword.set_active(rs['iw'])
            
                
    
    def _search_key_press(self,widget,event,dialog):
        sym=event.keyval
        if sym == Gdk.KEY_Escape:
            dialog.response(Gtk.ResponseType.CANCEL)
            return 1
        elif sym in (Gdk.KEY_Return, Gdk.KEY_KP_Enter):
            if not dialog.is_replace_dialog:
                dialog.response(Gtk.ResponseType.OK)
                return 1
        elif sym == Gdk.KEY_Up:
            self._search_history(dialog,True)
            return 1
        elif sym == Gdk.KEY_Down:
            self._search_history(dialog,False)
            return 1
        return 0
        
    def _create_find_dialog(self,is_replace):
        dialog=Gtk.Dialog(Gtk.STOCK_FIND,self,3)
        dialog.add_button(Gtk.STOCK_CANCEL,Gtk.ResponseType.CANCEL)
        dialog.add_button(Gtk.STOCK_OK,Gtk.ResponseType.OK)
        inn=dialog.get_content_area()
        grid=Gtk.Grid()
        inn.pack_start(grid,0,0,2)
        grid.attach(Gtk.Label('Search'),0,0,1,1)
        dialog.find_entry=Gtk.Entry()
        dialog.find_entry.set_size_request(300,-1)
        grid.attach(dialog.find_entry,1,0,4,1)
        dialog.is_replace_dialog=is_replace
        dialog.history_number=0
        dialog.manual_form={
            'find':'',
            'replace':'',
            're':False,
            'cs':False,
            'sw':False,
            'ew':False,
            'iw':False
        }
        dialog.find_entry.connect("key-press-event",self._search_key_press,dialog);
        row=1
        dialog.cb_regexp=Gtk.CheckButton.new_with_label('Regexp')
        grid.attach(dialog.cb_regexp,0,row,1,1)
        dialog.cb_regexp.connect('toggled',self._rword,(0,dialog))
        dialog.cb_sensitive=Gtk.CheckButton.new_with_label('Case sensitive')
        grid.attach(dialog.cb_sensitive,1,row,1,1)
        dialog.cb_startword=Gtk.CheckButton.new_with_label('Starts word')
        grid.attach(dialog.cb_startword,2,row,1,1)
        dialog.cb_startword.connect('toggled',self._rword,(1,dialog))
        dialog.cb_endword=Gtk.CheckButton.new_with_label('Ends word')
        grid.attach(dialog.cb_endword,3,row,1,1)
        dialog.cb_endword.connect('toggled',self._rword,(2,dialog))
        dialog.cb_isword=Gtk.CheckButton.new_with_label('Whole word')
        grid.attach(dialog.cb_isword,4,row,1,1)
        dialog.cb_isword.connect('toggled',self._rword,(3,dialog))
        row += 1
        if not is_replace:
            dialog.set_default_response(Gtk.ResponseType.OK)
            dialog.cb_backward=Gtk.CheckButton.new_with_label('Search backwards')
            grid.attach(dialog.cb_backward,0,row,2,1)
        inn.show_all()
        return dialog
        
    
    def _apply_font(self):
        if self.font:
            font_descr=Pango.font_description_from_string(self.font)
            self.view.modify_font(font_descr)
            r=re.match(r'.*\s([0-9]+)$',self.font)
            if r:
                n=int(r.group(1)) /2
                self.view.set_pixels_below_lines(n)
                
            
            
    
    def _select_font(self,*args):
        if self.speaking:
            return
        dialog=Gtk.FontChooserDialog('Select font',self)
        if self.font:
            dialog.set_font(self.font)
        i1=self.buffer.get_start_iter()
        i2=self.buffer.get_iter_at_offset(80)
        txt=self.buffer.get_text(i1,i2,False).strip()
        txt=re.sub(r'\s+',' ',txt)
        if len(txt) > 16:
            dialog.set_preview_text(txt)
        r=dialog.run()
        if r == Gtk.ResponseType.OK:
            self.font=dialog.get_font()
            self._apply_font()
            self.core.modify_params({'font':self.font})
        dialog.destroy()
    
    
    def _translate(self,*args):
        if self.speaking:
            return
        txt=''
        sel=self.buffer.get_selection_bounds()
        if sel:
            txt=self.buffer.get_text(sel[0],sel[1],False)
        elif self.last_sentence:
            txt=self.last_sentence
            self.last_sentence=None
        else:
            txt=self._last_txt
        langs=self.core.translator_languages()
        if not langs:
            return
        dialog=Gtk.Dialog('Translator',self,3)
        dialog.add_button(Gtk.STOCK_CANCEL,0)
        dialog.add_button(Gtk.STOCK_OK,1)
        inn=dialog.get_content_area()
        hb=Gtk.Box(Gtk.Orientation.HORIZONTAL)
        inn.pack_start(hb,0,0,2)
        entry=Gtk.Entry()
        entry.set_text(txt)
        entry.set_size_request(750,-1);
        hb.pack_start(entry,0,0,2)
        hb=Gtk.Box(Gtk.Orientation.HORIZONTAL)
        inn.pack_start(hb,0,0,2)
        lfr=Gtk.ComboBoxText()
        for a in langs:
            lfr.append_text(a[0])
        hb.pack_start(lfr,0,0,2) 
        lto=Gtk.ComboBoxText()
        for a in langs:
            lto.append_text(a[0])
        hb.pack_start(Gtk.Label('=>'),0,0,2) 
        hb.pack_start(lto,0,0,2)
        l1=0
        l2=0
        for n,a in enumerate(langs):
            if a[1] == self.llang_f:
                l1=n
            if a[1] == self.llang_t:
                l2=n
        lfr.set_active(l1)
        lto.set_active(l2)
        inn.show_all()
        r=dialog.run()
        txt=entry.get_text()
        l1=lfr.get_active()
        l2=lto.get_active()
        dialog.destroy()
        if r != 1:
            return None
        self.llang_f=langs[l1][1]
        self.llang_t=langs[l2][1]
        self.core.modify_params({'langf':self.llang_f,'langt':self.llang_t})
        if not txt:
            return None
        self._last_txt=txt
        txt=self.core.translate(txt,self.llang_f,self.llang_t)
        return txt
        
    
    
    def _save_file_dialog(self,*args):
        if self.speaking:
            return None
        dialog=Gtk.FileChooserDialog('Open file',self,Gtk.FileChooserAction.OPEN,
                                     (Gtk.STOCK_CANCEL,Gtk.ResponseType.CANCEL,Gtk.STOCK_OK,Gtk.ResponseType.OK))
        dialog.set_local_only(True)
        if self._last_file:
            dialog.set_filename(self._last_file)
        else:
            dialog.set_current_folder(self._last_dir)
        dialog.set_do_overwrite_confirmation(True)
        dialog.set_create_folders(True)
        f1=Gtk.FileFilter()
        f1.set_name('Text files')
        f1.add_pattern('*.txt')
        dialog.add_filter(f1)
        f2=Gtk.FileFilter()
        f2.set_name('All files')
        f2.add_pattern('*')
        dialog.add_filter(f2)
        dialog.set_filter(f1)
        rc=dialog.run()
        fname=None
        if rc == Gtk.ResponseType.OK:
            fname=dialog.get_filename()
            self._last_dir=dialog.get_current_folder()
        dialog.destroy()
        return fname
    
    def _save_file(self,*args):
        if not self._last_file:
            self._save_file_as()
            return
        if self.core.save_file(self._last_file,self.buffer.get_text(
                self.buffer.get_start_iter(),self.buffer.get_end_iter(),False)):
            self.buffer.set_modified(False)
            self.core.enable_bookmarks()
            self._show_bookmarks()
    
    def _save_file_as(self,*args):
        fname=self._save_file_dialog()
        if not fname:
            return
        if self.core.save_file(fname,self.buffer.get_text(
                self.buffer.get_start_iter(),self.buffer.get_end_iter(),False)):
            self._last_file=fname
            self.buffer.set_modified(False)
            self.core.enable_bookmarks()
            self._show_bookmarks()
            self.set_title(os.path.basename(fname))
    
    
    def _open_file(self,*args):
        if self.speaking:
            return
        dialog=Gtk.FileChooserDialog('Open file',self,Gtk.FileChooserAction.OPEN,
                                     (Gtk.STOCK_CANCEL,Gtk.ResponseType.CANCEL,Gtk.STOCK_OK,Gtk.ResponseType.OK))
        dialog.set_local_only(True)
        dialog.set_current_folder(self._last_dir)
        f1=Gtk.FileFilter()
        f1.set_name('Text files')
        f1.add_pattern('*.txt')
        dialog.add_filter(f1)
        f2=Gtk.FileFilter()
        f2.set_name('All files')
        f2.add_pattern('*')
        dialog.add_filter(f2)
        dialog.set_filter(f1)
        rc=dialog.run()
        fname=None
        if rc == Gtk.ResponseType.OK:
            fname=dialog.get_filename()
            self._last_dir=dialog.get_current_folder()
        dialog.destroy()
        if (fname):
            self.core.set_file(fname)
    
    def _set_speaking_buttons(self):
        for a in self.list_speak:
            a.set_sensitive(self.speaking)
        for a in self.list_unspeak:
            a.set_sensitive(not self.speaking)
        t=(not self.speaking) and self.editable
        for a in self.list_editonly:
            a.set_sensitive(t)
        
        self.view.set_editable(self.editable and not self.speaking)

    def _speak_start(self,*args):
        if self.speaking:
            return
        lang=self.voices[self.voicecombo.get_active()][1]
        vox=self.voices[self.voicecombo.get_active()][0]
        speed=self.voicespeed.get_value()
        self.core.modify_params({
            'voice':vox,
            'speed':speed,
            lang+'_voice':vox,
            lang+'_speed':speed})
        tm=self.buffer.get_insert()
        itr1=self.buffer.get_iter_at_mark(tm)
        itr2=self.buffer.get_end_iter()
        offset=itr1.get_offset()
        txt=self.buffer.get_text(itr1,itr2,False).decode('utf-8')
        if not self.core.init_speaker(vox,lang,offset,txt,speed):
            return
        self.speaking=True
        self._set_speaking_buttons()
        GLib.timeout_add(100,self._show_speak,None)
        self.last_cursor_pos=offset-1;
        self.position=None
        self.old_position=None
        self.core.start_speak()

    def _is_position_visible(self):
        r=self.view.get_visible_rect()
        t=self.view.get_iter_location(self.buffer.get_iter_at_offset(self.position[0]))
        if t.y < r.y:
            return True
        t=self.view.get_iter_location(self.buffer.get_iter_at_offset(self.position[0]+self.position[1]))
        return t.y+t.height > r.y+r.height
            
    
    def _show_speak(self, *args):
        if not self.speaking:
            self._set_speaking_buttons()
            if (self.old_position):
                i1=self.buffer.get_iter_at_offset(self.old_position[0])
                i2=self.buffer.get_iter_at_offset(self.old_position[0]+self.old_position[1])
                self.buffer.remove_tag(self.speaktag,i1,i2)
                self.last_sentence=self.buffer.get_text(i1,i2,0)
            else:
                self.last_sentence=None
            self.start_button.set_sensitive(True)
            self.stop_button.set_sensitive(False)
            return False
        if self.position and self.position[0] != self.last_cursor_pos:
            self.last_cursor_pos=self.position[0]
            itr=self.buffer.get_iter_at_offset(self.last_cursor_pos)
            if self._is_position_visible():
                self.view.scroll_to_iter(itr,0.05,True,0.1,0.0)
            self.buffer.place_cursor(itr)
            if (self.old_position):
                self.buffer.remove_tag(self.speaktag,
                                       self.buffer.get_iter_at_offset(self.old_position[0]),
                                       self.buffer.get_iter_at_offset(self.old_position[0]+self.old_position[1]))
            self.old_position=self.position
            self.buffer.apply_tag(self.speaktag,
                                       self.buffer.get_iter_at_offset(self.old_position[0]),
                                       self.buffer.get_iter_at_offset(self.old_position[0]+self.old_position[1]))
            
            
        return True
    
    def _clauth(self, *args):
        if not self.ask('Clear MS client id and secret?'):
            return
        self.core.translator_forget()
    
    def _speak_stop(self,*args):
        self.core.stop_speak()
        self.speaking=False
    
    def _set_editable(self,*args):
        self.editable=self.edit_button.get_active()
        self.view.set_editable(self.editable and not self.speaking)
        if self.editable:
            self.core.disable_bookmarks()
        elif not self.buffer.get_modified():
            self.core.enable_bookmarks()
        self._show_bookmarks()
        self._set_speaking_buttons()
        
    def _show_bookmarks(self):
        if self.editable:
            bm=False
        else:
            bm=self.core.bookmarks_enabled
        self.bookmark_indicator.set_from_stock(Gtk.STOCK_YES if bm else Gtk.STOCK_NO,Gtk.IconSize.BUTTON)
        self.bookmark_indicator.set_tooltip_text("Bookmark saving enabled" if bm else "Bookmark saving disabled")
            
    
    def __init__(self,core):
        Gtk.Window.__init__(self)
        self.core=core
        self.error=None
        self._last_txt=''
        self._last_dir=os.getcwd()
        self._last_file=None
        self.last_sentence=None
        params=self.core.get_params(['font','langf','langt','voice','speed'])
        self.font=params['font']
        self.llang_f=params['langf']
        self.llang_t=params['langt']
        self.speaking=False
        self.editable=False
        self.list_unspeak=[]
        self.list_speak=[]
        self.list_editonly=[]
        self.findlist=[]
        
        
        self.cid=self.connect("destroy",self.exit)
        frame=Gtk.VBox(0,2)
        self.add(frame)
        
        accel=Gtk.AccelGroup()
        self.add_accel_group (accel)

        menubar=Gtk.MenuBar()
        filemenu=Gtk.Menu()
        editmenu=Gtk.Menu()
        speechmenu=Gtk.Menu()
        transmenu=Gtk.Menu()
        
        m_file=Gtk.MenuItem.new_with_mnemonic('_File')
        m_file.set_submenu(filemenu)
        m_open=Gtk.ImageMenuItem.new_from_stock(Gtk.STOCK_OPEN,None)
        m_open.add_accelerator('activate',accel,Gdk.KEY_o,Gdk.ModifierType.CONTROL_MASK,Gtk.AccelFlags.VISIBLE)
        m_open.connect('activate',self._open_file,None)
        self.list_unspeak.append(m_open)
        m_save=Gtk.ImageMenuItem.new_from_stock(Gtk.STOCK_SAVE,None)
        m_save.add_accelerator('activate',accel,Gdk.KEY_s,Gdk.ModifierType.CONTROL_MASK,Gtk.AccelFlags.VISIBLE)
        m_save.connect('activate',self._save_file,None)
        self.list_unspeak.append(m_save)

        m_savea=Gtk.ImageMenuItem.new_from_stock(Gtk.STOCK_SAVE_AS,None)
        m_savea.add_accelerator('activate',accel,Gdk.KEY_s,Gdk.ModifierType.CONTROL_MASK | Gdk.ModifierType.SHIFT_MASK,Gtk.AccelFlags.VISIBLE)
        m_savea.connect('activate',self._save_file_as,None)
        self.list_unspeak.append(m_savea)
        
        
        m_font=Gtk.ImageMenuItem.new_from_stock(Gtk.STOCK_SELECT_FONT,None)
        m_font.connect('activate',self._select_font,None)
        self.list_unspeak.append(m_font)
        m_quit=Gtk.ImageMenuItem.new_from_stock(Gtk.STOCK_QUIT,None)
        m_quit.add_accelerator('activate',accel,Gdk.KEY_q,Gdk.ModifierType.CONTROL_MASK,Gtk.AccelFlags.VISIBLE)
        m_quit.connect('activate',self.exit,None)
        filemenu.append(m_open)
        filemenu.append(m_save)
        filemenu.append(m_savea)
        filemenu.append(m_font)
        filemenu.append(m_quit)
        menubar.append(m_file)
        
        m_edit=Gtk.MenuItem.new_with_mnemonic('_Edit')
        m_edit.set_submenu(editmenu)
        m_find=Gtk.ImageMenuItem.new_from_stock(Gtk.STOCK_FIND,None)
        m_find.add_accelerator('activate',accel,Gdk.KEY_f,Gdk.ModifierType.CONTROL_MASK,Gtk.AccelFlags.VISIBLE)
        m_find.connect('activate',self._find_dialog,None)
        self.list_unspeak.append(m_find)

        m_findn=Gtk.MenuItem('Find next')
        m_findn.add_accelerator('activate',accel,Gdk.KEY_g,Gdk.ModifierType.CONTROL_MASK,Gtk.AccelFlags.VISIBLE)
        m_findn.connect('activate',self._find_next,None)
        self.list_unspeak.append(m_findn)

        m_findp=Gtk.MenuItem('Find previous')
        m_findp.add_accelerator('activate',accel,Gdk.KEY_g,Gdk.ModifierType.CONTROL_MASK | Gdk.ModifierType.SHIFT_MASK,Gtk.AccelFlags.VISIBLE)
        m_findp.connect('activate',self._find_previous,None)
        self.list_unspeak.append(m_findp)

        editmenu.append(m_find)
        editmenu.append(m_findn)
        editmenu.append(m_findp)
        menubar.append(m_edit)

        m_speech=Gtk.MenuItem.new_with_mnemonic('_Speech')
        m_speech.set_submenu(speechmenu)
        m_play=Gtk.ImageMenuItem.new_from_stock(Gtk.STOCK_MEDIA_PLAY,None)
        m_play.add_accelerator('activate',accel,Gdk.KEY_p,Gdk.ModifierType.CONTROL_MASK,Gtk.AccelFlags.VISIBLE)
        m_play.connect('activate',self._speak_start,None)
        self.list_unspeak.append(m_play)
        m_stop=Gtk.ImageMenuItem.new_from_stock(Gtk.STOCK_MEDIA_STOP,None)
        m_stop.add_accelerator('activate',accel,Gdk.KEY_Escape,0,Gtk.AccelFlags.VISIBLE)
        m_stop.connect('activate',self._speak_stop,None)
        self.list_speak.append(m_stop)
        speechmenu.append(m_play)
        speechmenu.append(m_stop)
        menubar.append(m_speech)

        m_trans=Gtk.MenuItem.new_with_mnemonic('T_ranslator')
        m_trans.set_submenu(transmenu)
        m_tran=Gtk.MenuItem('Translate')
        m_tran.add_accelerator('activate',accel,Gdk.KEY_t,Gdk.ModifierType.CONTROL_MASK,Gtk.AccelFlags.VISIBLE)
        m_tran.connect('activate',self._translate,None)
        self.list_unspeak.append(m_tran)
        m_tcln=Gtk.MenuItem('Clean auth data')
        m_tcln.connect('activate',self._clauth,None)
        self.list_unspeak.append(m_tcln)
        transmenu.append(m_tran)
        transmenu.append(m_tcln)
        menubar.append(m_trans)

        frame.pack_start(menubar,0,0,3)
        
        self.toolbar=Gtk.Toolbar()
        self.toolbar.set_style(Gtk.ToolbarStyle.BOTH)
        frame.pack_start(self.toolbar,0,0,0)
        
        button=Gtk.ToolButton.new_from_stock(Gtk.STOCK_OPEN)
        self.toolbar.insert(button,-1)
        button.connect('clicked',self._open_file,None)
        self.list_unspeak.append(button)
        
        self.toolbar.insert(Gtk.SeparatorToolItem(),-1)

        self.start_button=Gtk.ToolButton.new_from_stock(Gtk.STOCK_MEDIA_PLAY)
        self.toolbar.insert(self.start_button,-1)
        self.start_button.connect('clicked',self._speak_start,None)
        self.stop_button=Gtk.ToolButton.new_from_stock(Gtk.STOCK_MEDIA_STOP)
        self.toolbar.insert(self.stop_button,-1)
        self.stop_button.connect('clicked',self._speak_stop,None)
        self.stop_button.set_sensitive(False)
        self.toolbar.insert(Gtk.SeparatorToolItem(),-1)
        self.list_unspeak.append(self.start_button)
        self.list_speak.append(self.stop_button)
        
        
        button=Gtk.ToolButton.new_from_stock(Gtk.STOCK_SELECT_FONT)
        button.connect('clicked',self._select_font,None)
        self.toolbar.insert(button,-1)
        self.list_unspeak.append(button)


        self.edit_button=Gtk.ToggleToolButton.new_from_stock(Gtk.STOCK_EDIT)
        self.edit_button.connect('toggled',self._set_editable,None)
        self.toolbar.insert(self.edit_button,-1)
        self.list_unspeak.append(self.edit_button)

        button=Gtk.ToolButton.new_from_stock(Gtk.STOCK_FIND)
        button.connect('clicked',self._find_dialog,None)
        self.toolbar.insert(button,-1)
        #self.list_editonly.append(button)
        self.list_unspeak.append(button)


        self.toolbar.insert(Gtk.SeparatorToolItem(),-1)

        self.translate_button=Gtk.ToolButton.new_from_stock(Gtk.STOCK_NETWORK)
        self.translate_button.set_label('Translator')
        self.toolbar.insert(self.translate_button,-1)
        self.translate_button.connect('clicked',self._translate,None)
        self.list_unspeak.append(self.translate_button)
        
        #voice selection
        self.voicebox=Gtk.Box(Gtk.Orientation.HORIZONTAL)
        frame.pack_start(self.voicebox,0,0,2)


        self.bookmark_indicator=Gtk.Image.new_from_stock(Gtk.STOCK_NO,Gtk.IconSize.BUTTON)
        self.voicebox.pack_start(self.bookmark_indicator,0,0,2)


        self.voicecombo=Gtk.ComboBoxText()
        self.voicebox.pack_start(self.voicecombo,0,0,2)
        self.voices=self.core.get_voices()
        n=0
        for i,vox in enumerate(self.voices):
            self.voicecombo.append_text(vox[0]+' ('+vox[1]+')')
            if vox[0] == params['voice']:
                n=i
        self.voicecombo.set_active(n)
        self.list_unspeak.append(self.voicecombo)

        self.voicespeed=Gtk.SpinButton.new_with_range(0.5,2.0,0.05)
        self.voicespeed.set_value(params['speed'] or 1.0)
        self.voicebox.pack_start(self.voicespeed,0,0,2)
        self.list_unspeak.append(self.voicespeed)
        

        
        self.resize(900,600)
        
        self.buffer=GtkSource.Buffer()
        self.view=GtkSource.View.new_with_buffer(self.buffer)
        self.view.set_show_line_numbers(True)
        self.view.set_highlight_current_line(True)
        self.view.set_size_request(750, 240);
        self.view.set_left_margin(5)
        self.view.set_right_margin(5)
        self.view.set_indent(15)
        self.view.set_editable(False)
        self.view.set_pixels_below_lines(4)
        self.view.set_wrap_mode(Gtk.WrapMode.WORD_CHAR)
        self.view.set_justification(Gtk.Justification.FILL)
        
        self.speaktag=self.buffer.create_tag('speaking')
        self.speaktag.set_property('background','yellow')
        self.speaktag.set_property('foreground','black')

        
        sw=Gtk.ScrolledWindow()
        sw.add(self.view)
        
        
        frame.pack_start(sw,True,True,0)

        self._apply_font()
        self.show_all()
        self.view.grab_focus()
        self._cursor=0
        self._show_bookmarks()
        GObject.threads_init()
        
    def _init_cursor(self, *args):
        itr=self.buffer.get_iter_at_offset(self._cursor)
        self.buffer.place_cursor(itr)
        tm=self.buffer.get_insert()
        self.view.scroll_to_mark(tm,0.05,True,0.1,0.0)
        #self.buffer.place_cursor(itr)
        return False
