#!/usr/bin/env python
#coding: utf-8

# sbreader.py - main class for SAPI Book Reader
# Copyright (C) Bohdan R. Rau 2013 <ethanak@polip.com>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this program.  If not, write to:
# 	The Free Software Foundation, Inc.,
# 	51 Franklin Street, Fifth Floor
# 	Boston, MA  02110-1301, USA.



from lib import gui,translator,nlp
import time
import sys,traceback,os,pickle,sapilektor,pyspda,threading,hashlib

#sudo easy_install guess-language

try:
    import guess_language
    language_guesser=True
except:
    language_guesser=False


class Core(object):
    def __init__(self):
        self._params={}
        self._config_name=os.path.join(os.environ['HOME'],'.sbreader.cfg')
        try:
            self._params=pickle.load(open(self._config_name))
        except:
            pass
        self._voices=None
        self.speaker=sapilektor.Lektor()
        self.translator=None
        self.speaking=False
        self.playing=False
        self.eob=False
        self.bookname=None
        self.bookmarks_enabled=False
        self.bmpath=None
        self.gui=gui.gui(self)
        self.pyspda=pyspda.Spda(1)
        self.lock=threading.Lock()
        self.page_thread=threading.Thread(target=self.pager)
        self.retr_thread=threading.Thread(target=self.retriever)
        self.play_thread=threading.Thread(target=self.player)
        self.page_thread.start()
        self.retr_thread.start()
        self.play_thread.start()
        
    def pager(self, *args):
        while True:
            if not self.speaking:
                time.sleep(0.1)
                continue
            if len(self.queue) >= 10:
                time.sleep(0.1)
                continue
            s=self.sentenser.get_sentence()
            if not s:
                self.eob=True
                time.sleep(0.1)
                continue
            self.lock.acquire()
            self.queue.append(s)
            self.lock.release()
    
    def retriever(self, *args):
        while True:
            if not self.speaking:
                time.sleep(0.1)
                continue;
            self.lock.acquire()
            txt=None
            for a in self.queue:
                if not a['wave']:
                    txt=a['txt']
                    pos=a['pos']
                    break
            self.lock.release()
            if not txt:
                time.sleep(0.1)
                continue
            wave=self.speaker.GetWave(txt.encode('utf-8'),True)
            self.lock.acquire()
            f=-1;
            for i,a in enumerate(self.queue):
                if a['pos']==pos:
                    if wave:
                        a['wave']=wave[0]
                        a['freq']=wave[1]
                    else:
                        f=i
                    break
            if f>=0:
                self.queue.pop(f)
            self.lock.release()
                        
            
    def player(self, *args):
        while True:
            if not self.speaking:
                if self.playing:
                    self.pyspda.finish()
                    self.playing=False
                    try:
                        self.save_bookmark(self.player_pos)
                    except:
                        traceback.print_exc()
                time.sleep(0.1)
                continue
            rec=None
            self.lock.acquire()
            if self.queue and len(self.queue)>0 and self.queue[0]['wave']:
                rec=self.queue.pop(0)
            self.lock.release()
            if not rec:
                if self.playing:
                    self.pyspda.finish()
                    self.playing=False
                if (not self.queue or len(self.queue)==0) and self.eob:
                    self.speaking=False
                    self.gui.end_of_book()
                    try:
                        self.save_bookmark(self.player_pos)
                    except:
                        traceback.print_exc()
                time.sleep(0.1)
                continue
            self.gui.set_play_position(rec['pos'],rec['count'])
            #print rec['txt']
            if not self.playing:
                self.playing=True
            self.player_pos=rec['pos']
            if rec['pause'] != nlp.ME_PAUSE_SENTENCE:
                try:
                    self.save_bookmark(rec['pos'])
                except:
                    traceback.print_exc()
            pau=nlp.ME_PAUSE_DEFAULTS[rec['pause']]
            if pau:
                self.pyspda.play(pau*44*'\0',freq=rec['freq'],rate=int(100*self.speed)-100)
                if not self.speaking:
                    if self.playing:
                        self.pyspda.finish()
                        self.playing=False
                    time.sleep(0.1)
                    continue
            self.pyspda.play(rec['wave'],freq=rec['freq'],rate=int(100*self.speed)-100)
            
    def query(self,what,title='Question'):
        return self.gui.query(what,title)
    
    def alert(self,text):
        self.gui.alert(text)
        
    def error(self,text):
        self.gui.alert(text,True)

    def disable_bookmarks(self):
        self.bookmarks_enabled=False

    def enable_bookmarks(self):
        self.bookmarks_enabled=True


    def get_params(self,params):
        rc={}
        for a in params:
            rc[a]=self._params.get(a)
        return rc
        
    
    def modify_params(self,params):
        mdf=False
        for a in params:
            if self._params.get(a) != params[a]:
                mdf=True
            self._params[a]=params[a]
        if mdf:
            pickle.dump(self._params,open(self._config_name,'w'))
        
    
    def set_text(self, text,n=0):
        text=text.replace('...','…')
        self.gui.set_text(text,n)
        
    
    def translator_forget(self):
        if self.translator:
            self.translator.forget()
        else:
            self.modify_params(
                {'ms_client_id':None,
                 'ms_client_secret':None}
            )

    
    def translator_languages(self):
        if not self.translator:
            self.translator=translator.Translator(self)
        try:
            return self.translator.langlist()
        except:
            self.error(traceback.format_exc())
            return None
        
            
    
    def translate(self,text,fl,tl):
        if not self.translator:
            self.translator=translator.Translator(self)
        try:
            r=self.translator.translate(text,fl,tl)
            if r:
                self.alert(r)
        except:
            self.error(traceback.format_exc())
    
    
    def read_bookmark(self):
        if not self.bmpath:
            return 0
        fname=os.path.join(os.environ['HOME'],'.sbreader',self.bmpath)
        try:
            return int(open(fname).read())
        except:
            return 0
    
    def save_bookmark(self,pos):
        if not self.bmpath:
            return
        dname=os.path.join(os.environ['HOME'],'.sbreader')
        if not os.path.isdir(dname):
            os.mkdir(dname)
        fname=os.path.join(os.environ['HOME'],'.sbreader',self.bmpath)
        open(fname,'w').write(str(pos))
    
    
    def set_file(self, filename):
        try:
            txt=open(filename).read()
        except:
            self.error(traceback.format_exc())
            return
        self.bookname=filename
        self.bmpath=hashlib.sha1(os.path.abspath(self.bookname)).hexdigest()
        n=self.read_bookmark()
        self.set_text(txt,n)
        self.gui.new_file_started(os.path.abspath(self.bookname))
        
    def save_file(self,filename,content):
        if os.path.isfile(filename):
            try:
                os.rename(filename,filename+'~')
            except OSError:
                self.error(sys.exc_info()[1].strerror)
                return False
            except:
                self.error(traceback.format_exc())
                return False
        try:
            open(filename,'w').write(content)
        except OSError:
            self.error(sys.exc_info()[1].strerror)
            return False
        except:
            self.error(traceback.format_exc())
            return False
        self.alert('File saved')
        return True
        
        
    def run(self):
        self.gui.loop()
    
    def get_voices(self):
        if self._voices:
            return self._voices
        voices=self.speaker.GetVoiceList()
        self._voices=[]
        for v in voices:
            self._voices.append([v[0],v[1]])
        self._voices.sort()
        return self._voices
    
    def init_speaker(self, voice, lang, offset, txt, speed):
        self.sentenser=nlp.sentenser()
        self.sentenser.feed(lang, offset, txt)
        try:
            self.speaker.SetSynthesisVoice(voice)
        except:
            self.error(traceback.format_exc())
            return False
        self.speed=speed
        self.spoken_sentence=None
        self.queue=[]
        self.eob=False
        for i in range(3):
            s=self.sentenser.get_sentence()
            if not s:
                self.eob=True
                break
            #wave=self.speaker.GetWave(txt.encode('utf-8'),True)
            #if wave:
            #    s['wave']=wave[0]
            #    s['freq']=wave[1]
            self.queue.append(s)
        return True
    
    def start_speak(self):
        self.pyspda.enable()
        self.speaking=True
    
    def stop_speak(self):
        self.speaking=False
        self.pyspda.stop()
    
    @staticmethod
    def language(txt):
        if not language_guesser:
            return None
        try:
            if not isinstance(txt,unicode):
                txt=txt.decode('utf-8')
            if len(txt) > 16000:
                txt=txt[:16000]
            lg=guess_language.guessLanguageInfo(txt)
            if not lg:
                return None
            return (lg[0],lg[2])
        except:
            return None

    
    

    

core=Core()
if len(sys.argv) > 1:
    core.set_file(sys.argv[1])
core.run()

    
    
    
