#!/usr/bin/env python
#coding: utf-8

# translator.py - Microsoft Translator class for SAPI Book Reader
# Copyright (C) Bohdan R. Rau 2013 <ethanak@polip.com>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this program.  If not, write to:
# 	The Free Software Foundation, Inc.,
# 	51 Franklin Street, Fifth Floor
# 	Boston, MA  02110-1301, USA.


import datetime,json,requests,locale


class Translator(object):
    oauth_url = 'https://datamarket.accesscontrol.windows.net/v2/OAuth2-13'
    translation_url = 'http://api.microsofttranslator.com/V2/Ajax.svc/Translate'
    service_url = 'http://api.microsofttranslator.com/V2/Ajax.svc/'
    
    def __init__(self,core):
        self.core=core
        self.client_id=None
        self.secret=None
        self.last_access=datetime.datetime.today()
        self.token=None
        self._langlist=None
        
    def forget(self):
        self.token=None
        self.client_id=None
        self.secret=None
        self.core.modify_params(
                {'ms_client_id':None,
                 'ms_client_secret':None}
            )
    
    def _get_token(self):
        d=datetime.datetime.today()
        if self.token and (d-self.last_access).total_seconds < 9 * 60:
            return True
        if not self.client_id or not self.secret:
            p=self.core.get_params(['ms_client_id','ms_client_secret'])
            self.client_id=p['ms_client_id']
            self.secret=p['ms_client_secret']
        modf=False
        if not self.client_id or not self.secret:
            p=self.core.query([['Client ID','ms_client_id'],['Client secret','ms_client_secret']],'Translator auth')
            if not p:
                return False
            self.client_id=p['ms_client_id']
            self.secret=p['ms_client_secret']
            modf=True
            
        params={
            'client_id':self.client_id,
            'client_secret':self.secret,
            'scope':'http://api.microsofttranslator.com',
            'grant_type':'client_credentials'
        }
        
        
        try:
            junk = json.loads(requests.post(self.__class__.oauth_url,data=params).content)
            self.token=junk['access_token']
        except KeyError:
            s='\n'.join(x+' '+junk[x] for x in junk)
            self.core.alert(s)
            self.client_id=None
            self.secret=None
            self.core.modify_params(
                {'ms_client_id':None,
                 'ms_client_secret':None}
            )
            return False
        except:
            import traceback
            self.core.alert(traceback.format_exc())
            return False
        self.last_access=datetime.datetime.today()
        if modf:
            self.core.modify_params(p)
        return True
        
    @staticmethod
    def lsort(a,b):
        if a[1]>b[1]:
            return 1
        if a[1]<b[1]:
            return -1
        a=a[0].encode('utf-8')
        b=b[0].encode('utf-8')
        return locale.strcoll(a,b)
    
            
    def langlist(self):
        if self._langlist:
            return self._langlist
        lc=locale.setlocale(locale.LC_COLLATE,'')
        if not self._get_token():
            return None
        headers={'Authorization': 'Bearer '+self.token}
        url=self.__class__.service_url+'GetLanguagesForTranslate'
        params={'appId':''}
        result = requests.get(url,params=params,headers=headers)
        llist=result.content.encode('utf-8')[3:]
        url=self.__class__.service_url+'GetLanguageNames'
        params={'appId':'','locale':lc[:2],'languageCodes':llist}
        result = requests.get(url,params=params,headers=headers)
        ulist=json.loads(result.content.encode('utf-8')[3:])
        llist=json.loads(llist)
        lista=[]
        while len(llist) > 0:
            a=ulist.pop(0)
            b=llist.pop(0)
            lista.append((a,b))
        lista.sort(self.lsort)
        self._langlist=tuple(lista)
        return self._langlist
    
    
    def translate(self,text,from_lang,to_lang):
        if not self._get_token():
            return None
        headers={'Authorization': 'Bearer '+self.token}
        params={'text':text,'from':from_lang,'to':to_lang}
        result = requests.get(self.__class__.translation_url,params=params,headers=headers)
        return json.loads(result.content.encode('utf-8')[3:])
